<?php
 /**
 * Jamroom Analytics module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2003 - 2023 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Get formatted session info to use in a tooltip
 * @param array $_session Session info
 * @return string
 */
function jrAnalytics_get_formatted_session_info($_session)
{
    $_out = array();
    foreach ($_session as $k => $v) {
        switch ($k) {
            case 'session_id':
            case 'session_sync':
                break;
            case 'session_updated':
                $_out[] = "{$k}: " . jrCore_format_time($v);
                break;
            default:
                $_out[] = "{$k}: {$v}";
                break;
        }
    }
    return jrCore_strip_non_utf8(implode('&#010;', $_out));
}

/**
 * get a formatted Geo Location
 * @param array $_info Geo Location
 * @return string
 */
function jrAnalytics_get_formatted_geo_location($_info)
{
    $_out = array();
    if (!empty($_info['country_name'])) {
        $_out[] = $_info['country_name'];
    }
    if (!empty($_info['region']) && !is_numeric($_info['region'])) {
        $_out[] = $_info['region'];
    }
    if (!empty($_info['city'])) {
        $_out[] = $_info['city'];
    }
    return jrCore_strip_non_utf8(implode(', ', $_out));
}

/**
 * Record a page view
 * @return bool
 */
function jrAnalytics_record_view()
{
    $cdr = jrCore_get_module_cache_dir('jrAnalytics');
    $ext = floor(time() / 60);
    $_tm = array(
        'p' => jrCore_get_ip(),
        'i' => (!empty($_SESSION['_user_id'])) ? intval($_SESSION['_user_id']) : 0,
        'u' => $_REQUEST['_uri'],
        'a' => (!empty($_SERVER['HTTP_USER_AGENT'])) ? $_SERVER['HTTP_USER_AGENT'] : ''
    );
    if ($domain = jrAnalytics_get_clean_referrer()) {
        $_tm['r'] = $domain;
    }
    return jrCore_write_to_file("{$cdr}/view.{$ext}", json_encode($_tm) . "\n", 'append');
}

/**
 * Process page views
 * @return void
 */
function jrAnalytics_process_page_views()
{
    $cdr = jrCore_get_module_cache_dir('jrAnalytics');
    if ($_files = glob("{$cdr}/view.*")) {
        $_pg = array();
        $_ur = array();
        foreach ($_files as $file) {
            list(, $date) = explode('.', basename($file));
            $date = date(($date * 60), 'Ymd');
            if ($_views = file($file)) {
                foreach ($_views as $view) {
                    // {"t":1679843612,"m":"GET","p":"127.0.0.1","i":1,"u":"\/pz\/dashboard\/pzprice\/browse\/profile_id=1","a":"Mozilla\/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit\/537.36 (KHTML, like Gecko) Chrome\/111.0.0.0 Safari\/537.36"}
                    if ($view = json_decode($view, true)) {
                        $_ur[] = "('" . jrCore_db_escape($view['u']) . "')";
                        $_pg[] = "({$date}," . intval($view['i']) . ",0,0)";
                    }
                }
            }
        }
    }
}

/**
 * Get a "clean" referrer - strip "www" and don't include local referrers
 * @return string|false
 */
function jrAnalytics_get_clean_referrer()
{
    if (!empty($_SERVER['HTTP_REFERER'])) {
        if ($domain = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_HOST)) {
            if (!jrCore_checktype($domain, 'ip_address')) {
                $domain = jrCore_str_to_lower(str_replace('www.', '', $domain));
                // Do we have a blocked keyword?
                if (!jrAnalytics_is_blocked_domain($domain)) {
                    // Don't count same domain referrers
                    if (!strpos(jrCore_get_base_url(), $domain)) {
                        $_data = jrCore_trigger_event('jrAnalytics', 'referring_domain', array('domain' => $domain));
                        if (!empty($_data['domain'])) {
                            return $domain;
                        }
                    }
                }
            }
        }
    }
    return false;
}

/**
 * Return TRUE if a given domain has a blocked keyword in it
 * @param string $domain Domain to check
 * @return bool
 */
function jrAnalytics_is_blocked_domain($domain)
{
    if ($_words = jrCore_get_config_value('jrAnalytics', 'blocked_referrers', false)) {
        if ($_words = explode("\n", $_words)) {
            foreach ($_words as $w) {
                $w = trim($w);
                if (!empty($w) && strpos(' ' . $domain, $w)) {
                    return true;
                }
            }
        }
    }
    return false;
}
