<?php
 /**
 * Jamroom Banned Items module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrBanned_meta()
{
    return array(
        'name'        => 'Banned Items',
        'url'         => 'banned',
        'version'     => '1.6.2',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Create, Update and Delete Banned names, words, email and IP addresses',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/1950/banned-items',
        'license'     => 'mpl',
        'requires'    => 'jrCore:6.5.12',
        'priority'    => 3,   // HIGH priority - we want to handle URLs before other modules
        'category'    => 'admin',
        'recommended' => 'jrThrottle'
    );
}

/**
 * init
 */
function jrBanned_init()
{
    // Check for banned IP addresses in process init
    jrCore_register_event_listener('jrCore', 'process_init', 'jrBanned_process_init_listener');
    jrCore_register_event_listener('jrCore', 'module_view', 'jrBanned_module_view_listener');
    jrCore_register_event_listener('jrCore', 'verify_module', 'jrBanned_verify_module_listener');

    // Block banned word searches
    jrCore_register_event_listener('jrSearch', 'search_fields', 'jrBanned_search_fields_listener');

    // We have a couple event triggers
    jrCore_register_event_trigger('jrBanned', 'banned_item_create', 'Fired with banned item info when a new item is created');
    jrCore_register_event_trigger('jrBanned', 'banned_item_delete', 'Fired with banned item info when an existing item is deleted');

    // Tool to create, update and delete banned Items
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrBanned', 'browse', array('Banned Items', 'Create, Update and Delete Banned names, words, email and IP addresses'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrBanned', 'test', array('Test Item', 'Test a value to see if it passes or fails the existing banned items'));

    jrCore_register_module_feature('jrCore', 'admin_tab', 'jrBanned', 'browse', 'Banned Items');
    jrCore_register_module_feature('jrCore', 'admin_tab', 'jrBanned', 'test', 'Test Items');

    // Our default master view
    jrCore_register_module_feature('jrCore', 'default_admin_view', 'jrBanned', 'browse');

    // System resets
    jrCore_register_event_listener('jrDeveloper', 'reset_system', 'jrBanned_reset_system_listener');

    return true;
}

//---------------------
// EVENT LISTENERS
//---------------------

/**
 * Cleanup schema on system reset
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrBanned_reset_system_listener($_data, $_user, $_conf, $_args, $event)
{
    $tbl = jrCore_db_table_name('jrBanned', 'banned');
    jrCore_db_query("TRUNCATE TABLE {$tbl}");
    jrCore_db_query("OPTIMIZE TABLE {$tbl}");
    return $_data;
}

/**
 * Check for banned IPs
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrBanned_process_init_listener($_data, $_user, $_conf, $_args, $event)
{
    if (jrCore_is_view_request()) {

        // make sure we ignore any worker calls from the cluster master
        if ($pfx = jrCore_get_flag('jrcloudclient_timer_prefix')) {
            if ($pfx == 'internal_') {
                return $_data;
            }
        }

        // Never block private IP addresses (10.x, 192.168.x, etc)
        $ip_address = jrCore_get_ip();
        if (jrCore_checktype($ip_address, 'private_ip_address')) {
            return $_data;
        }

        // Is this viewer already in Time out?
        if (jrCore_get_local_cache_key("bto:{$ip_address}", false)) {

            // yes - they are in time out
            jrCore_set_custom_header('HTTP/1.0 403 Forbidden', 'status_403');
            jrCore_notice('error', 'You do not have permission to access this server', false);

        }

        // Are we auto blocking bad URLs and blocked user agents?
        // @note: We check for URLs and agents BEFORE checking for
        // banned IP addresses since there is no DB lookup
        if (jrCore_get_config_value('jrBanned', 'auto_block', 'off') === 'on') {

            // empty user_agent should be rare
            if (!empty($_SERVER['HTTP_USER_AGENT'])) {
                if ($string = jrBanned_is_blocked_user_agent($_SERVER['HTTP_USER_AGENT'])) {

                    // We have a possible site scanner running on the site fishing for URLs
                    if (jrCore_get_config_value('jrBanned', 'log_block', 'on') == 'on') {
                        $url = (!empty($_REQUEST['_uri'])) ? "/{$_REQUEST['_uri']}" : '/';
                        jrCore_logger('MAJ', "banned: blocked agent: {$string} to URL: {$url}");
                    }

                    // Timeout this viewer for 60 minutes
                    jrCore_set_local_cache_key("bto:{$ip_address}", 1, 3600, false);
                    jrCore_set_custom_header('HTTP/1.0 403 Forbidden', 'status_403');
                    jrCore_notice('error', 'You do not have permission to access this server', false);

                }
            }

            // Watch for blocked URLs
            if (!empty($_REQUEST['_uri'])) {
                if ($string = jrBanned_is_honeypot_url($_REQUEST['_uri'])) {

                    // Are we auto banning?
                    $add_log = '';
                    $seconds = 300;
                    if (jrCore_get_config_value('jrBanned', 'auto_block_ban', 'on') == 'on') {

                        // Yes - we are being asked to auto-ban IP addresses of requests that hit our honeypot URLs
                        // Make sure this is not a master/admin users
                        $_user = jrUser_session_start(false);
                        if (!jrUser_is_admin()) {

                            // Ban...
                            if (jrUser_is_logged_in()) {
                                $txt = "auto_ban: {$_user['_user_id']}/{$_user['user_name']}/{$_user['user_email']} - by URL: {$_REQUEST['_uri']}";
                            }
                            else {
                                $txt = "auto_ban by URL: {$_REQUEST['_uri']}";
                            }
                            if (strlen($txt) > 4090) {
                                $txt = substr($txt, 0, 4087) . '...';
                            }
                            $txt = jrCore_db_escape($txt);
                            $tbl = jrCore_db_table_name('jrBanned', 'banned');
                            $req = "INSERT INTO {$tbl} (ban_updated, ban_type, ban_value, ban_note, ban_partial)
                                    VALUES (UNIX_TIMESTAMP(), 'ip', '{$ip_address}', '{$txt}', 'off')
                                    ON DUPLICATE KEY UPDATE ban_updated = UNIX_TIMESTAMP()";
                            $cnt = jrCore_db_query($req, 'COUNT');
                            if ($cnt === 1) {
                                // Reset caches
                                $key = 'jrbanned_config_ip';
                                jrCore_delete_flag($key);
                                jrCore_delete_local_cache_key($key);
                                jrCore_delete_cache('jrBanned', $key, false, false);

                                // Trigger our creation
                                $_REQUEST['ban_type'] = 'ip';
                                jrCore_trigger_event('jrBanned', 'banned_item_create', $_REQUEST);
                            }
                            $seconds = 86400;
                            $add_log = ' (auto banned)';
                        }
                    }

                    // Are we logging?
                    if (jrCore_get_config_value('jrBanned', 'log_block', 'on') == 'on') {
                        jrCore_logger('MAJ', "banned: blocked text: {$string} in URL: {$_REQUEST['_uri']}{$add_log}");
                    }

                    // Timeout this IP so we do not have to check again
                    jrCore_set_local_cache_key("bto:{$ip_address}", 1, $seconds, false);
                    jrCore_set_custom_header('HTTP/1.0 403 Forbidden', 'status_403');
                    jrCore_notice('error', 'You do not have permission to access this server', false);

                }
            }
        }

        // Make sure this is NOT a banned IP
        if (jrBanned_is_banned('ip', $ip_address)) {
            // Timeout this IP for 3600 seconds
            jrCore_set_local_cache_key("bto:{$ip_address}", 1, 3600, false);
            jrCore_set_custom_header('HTTP/1.0 403 Forbidden', 'status_403');
            jrCore_notice('error', 'You do not have permission to access this server', false);
        }
    }

    return $_data;
}

/**
 * Check for banned words in form submissions
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrBanned_module_view_listener($_data, $_user, $_conf, $_args, $event)
{
    // Watch for form submissions
    if (jrCore_get_config_value('jrBanned', 'form_scan', 'on') == 'on') {
        if (!jrUser_is_admin() && isset($_data['module']) && $_data['module'] == 'jrCore' && isset($_data['option']) && $_data['option'] == 'form_validate' && jrCore_is_ajax_request()) {
            if (isset($_data['jr_html_form_token'])) {
                if ($_rt = jrCore_form_get_session($_data['jr_html_form_token'])) {
                    // Scan for incoming text fields (text, textarea) and check for banned words
                    if (isset($_rt['form_fields']) && is_array($_rt['form_fields'])) {
                        foreach ($_rt['form_fields'] as $_field) {
                            if (isset($_field['ban_check'])) {
                                if ($_field['ban_check'] === false) {
                                    // Purposefully disabled on this field
                                    continue;
                                }
                            }
                            $key  = false;
                            $type = false;
                            switch ($_field['type']) {
                                case 'editor':
                                    $key  = "{$_field['name']}_editor_contents";
                                    $type = 'word';
                                    break;
                                case 'select_and_text':
                                    $key  = "{$_field['name']}_text";
                                    $type = 'word';
                                    break;
                                case 'text':
                                    $key = $_field['name'];
                                    if (strpos($key, '_name')) {
                                        $type = 'name';
                                    }
                                    elseif (strpos($key, '_email')) {
                                        $type = 'email';
                                    }
                                    else {
                                        $type = 'word';
                                    }
                                    break;
                                case 'textarea':
                                    $key  = $_field['name'];
                                    $type = 'word';
                                    break;
                            }
                            if ($key && $type && isset($_data[$key]) && strlen($_data[$key]) > 0) {
                                $check = true;
                                switch ($key) {
                                    case 'profile_name':
                                    case 'user_name':
                                    case 'user_email':
                                        if ($_data[$key] == $_user[$key]) {
                                            $check = false;
                                        }
                                        break;
                                }
                                if ($check) {
                                    if ($bad = jrBanned_is_banned($type, $_data[$key])) {
                                        $_ln = jrUser_load_lang_strings();
                                        jrCore_set_form_notice('error', "{$_ln['jrBanned'][1]}&quot;{$bad}&quot;");
                                        jrCore_form_field_hilight($key);
                                        jrCore_form_result();
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return $_data;
}

/**
 * Migrate from jrCore to jrBanned
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrBanned_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // See if we have migrated
    if (jrCore_db_table_exists('jrCore', 'banned')) {
        // We have not migrated yet
        $tb1 = jrCore_db_table_name('jrCore', 'banned');
        $tb2 = jrCore_db_table_name('jrBanned', 'banned');
        $req = "INSERT INTO {$tb2} SELECT * FROM {$tb1}";
        jrCore_db_query($req);
        $req = "DROP TABLE {$tb1}";
        jrCore_db_query($req);
    }
    return $_data;
}

/**
 * Block banned word searches
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrBanned_search_fields_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (!jrUser_is_admin() && isset($_post['search_string']) && strlen($_post['search_string']) > 0) {
        $_tm = explode(' ', $_post['search_string']);
        if ($_tm && is_array($_tm)) {
            foreach ($_tm as $word) {
                if (jrBanned_is_banned('word', $word)) {
                    // This is a banned word - short circuit our match
                    return array(
                        'jrBanned' => array(
                            'no_match' => 1
                        )
                    );
                }
            }
        }
    }
    return $_data;
}

//---------------------
// FUNCTIONS
//---------------------

/**
 * Get the HELP output for entering a new banned item
 * @return string
 */
function jrBanned_get_type_help()
{
    $out = 'Enter the values to match for Banned Items. Each entry should be on a separate line.<br><br><b>IP Address</b> - Enter a valid IP Address or partial IP Address to block all system requests to your system from the IP. For example, use &quot;123.125.&quot; to ban the IP range 123.125.0.0 to 123.125.255.255.<br><br><b>Profile or User Name</b> - Enter words that cannot be used in a User Name, Profile Name or Profile URL.<br><br><b>Email Address or Domain</b> - An Email address - to block an entire email domain, enter just the domain name and enable <b>Match Substring</b> below - i.e. @example.com.<br><br><b>Forbidden Word</b> - Any word entered as a forbidden word will not be allowed in items created by a User.';

    // Do we have other registered types?
    $_mf = jrCore_get_registered_module_features('jrBanned', 'banned_type');
    if ($_mf && is_array($_mf)) {
        foreach ($_mf as $_inf) {
            foreach ($_inf as $_data) {
                if (!empty($_data['help'])) {
                    $out .= "<br><br><b>{$_data['title']}</b> - {$_data['help']}";
                }
            }
        }
    }
    return $out;
}

/**
 * Get list of honeypot URLs for auto blocking
 * @return array
 */
function jrBanned_get_default_honeypot_urls()
{
    return array(
        '/../',
        '.git/',
        '.env',
        'bxss.me',
        '.chr(',
        '.concat(',
        '.gethostbyname(',
        '(nslookup',
        '__import__',
        '/webdav',
        '/wp-login.php',
        '/wp-admin.php',
        '/wp-admins.php',
        '/wp-config.php',
        '/eval-stdin.php',
        '/wp-includes',
        '/wp-content',
        '/xmlrpc.php',
        '/azenv.php',
        '/webadmin.php',
        '/dbadmin',
        '/myadmin',
        '/pma',
        '/mysqlmanager',
        '/mysql-admin',
        '/webconfig',
        '/sqlmanager',
        'TP/public',
        '/cgi-bin',
        '/cgi-perl',
        '/cgi-exe',
        '%20union%20select'
    );
}

/**
 * Get list of honeypot URLs for auto blocking
 * @return array|false
 */
function jrBanned_get_honeypot_urls()
{
    if ($urls = jrCore_get_config_value('jrBanned', 'auto_block_strings', false)) {
        return explode("\n", trim($urls));
    }
    return false;
}

/**
 * Return TRUE if requested URL is a honeypot URL
 * @param string $url
 * @return mixed string|bool
 */
function jrBanned_is_honeypot_url($url)
{
    if ($_urls = jrBanned_get_honeypot_urls()) {
        foreach ($_urls as $u) {
            if (stripos(' ' . $url, trim($u))) {
                return $u;
            }
        }
    }
    return false;
}

/**
 * Get array of strings to check in User Agent
 * @return array
 */
function jrBanned_get_default_blocked_user_agent_strings()
{
    return array(
        'masscan',
        'jorgee',
        'nikto',
        'zmeu',
        'nmap.org',
        'bxss.me'
    );
}

/**
 * Get list of honeypot URLs for auto blocking
 * @return array|false
 */
function jrBanned_get_blocked_user_agent_strings()
{
    if ($temp = jrCore_get_config_value('jrBanned', 'auto_block_agents', false)) {
        return explode("\n", trim($temp));
    }
    return false;
}

/**
 * Return TRUE if given User Agent contains a blocked User Agent string
 * @param string $agent
 * @return mixed string|bool
 */
function jrBanned_is_blocked_user_agent($agent)
{
    if ($_strings = jrBanned_get_blocked_user_agent_strings()) {
        foreach ($_strings as $s) {
            if (stripos(' ' . $agent, trim($s))) {
                return $s;
            }
        }
    }
    return false;
}

/**
 * Get current banned config for a given ban type
 * @param $type string
 * @return array|false
 */
function jrBanned_get_banned_config($type)
{
    $key = "jrbanned_config_{$type}";
    if (!$_rt = jrCore_get_flag($key)) {
        if (!$_rt = jrCore_get_local_cache_key($key)) {
            if (!$_rt = jrCore_is_cached('jrBanned', $key, false, false)) {
                $tbl = jrCore_db_table_name('jrBanned', 'banned');
                $req = "SELECT ban_id AS i, ban_value AS v, ban_partial AS p FROM {$tbl} WHERE ban_type = '" . jrCore_db_escape($type) . "'";
                $_xt = jrCore_db_query($req, 'NUMERIC', false, null, false);
                if (!$_xt || !is_array($_xt)) {
                    $_rt = 'no_items';
                }
                else {
                    $_rt = array();
                    foreach ($_xt as $xt) {
                        $xt['p']                     = (isset($xt['p'])) ? $xt['p'] : '';
                        $_rt["{$xt['i']}{$xt['p']}"] = $xt['v'];
                    }
                }
                jrCore_set_flag($key, $_rt);
                jrCore_set_local_cache_key($key, $_rt);
                jrCore_add_to_cache('jrBanned', $key, $_rt, 0, 0, false, false);
            }
            else {
                jrCore_set_flag($key, $_rt);
                jrCore_set_local_cache_key($key, $_rt);
            }
        }
        else {
            jrCore_set_flag($key, $_rt);
        }
    }
    if (!$_rt || !is_array($_rt) || count($_rt) === 0) {
        // No items of this type
        return false;
    }
    return $_rt;
}

/**
 * Test if a given value for a type is a banned item
 * @param string $type Type of Banned Item
 * @param string $value Value to check
 * @return bool
 */
function jrBanned_is_banned($type, $value)
{
    if (mb_strlen($value) > 1) {
        if (!$_rt = jrBanned_get_banned_config($type)) {
            // No items of this type
            return false;
        }
        switch ($type) {

            case 'ip':
                $value = trim($value);
                if (!empty($_SERVER['SERVER_ADDR']) && $_SERVER['SERVER_ADDR'] == $value) {
                    // Do not allow the local server IP to be banned
                    return false;
                }
                foreach ($_rt as $v) {
                    if (strpos($value, $v) === 0) {
                        return $v;
                    }
                }
                break;

            case 'word':
            case 'name':
            case 'email':
                $value = trim(strip_tags($value));
                foreach ($_rt as $i => $v) {
                    if (strpos($i, 'on')) {
                        if (stripos($value, $v) === 0 || stripos($value, $v)) {
                            return $v;
                        }
                    }
                    else {
                        if ($v == $value || preg_match('/\b' . preg_quote($v) . '\b/ui', " {$value} ")) {
                            return $v;
                        }
                    }
                }
                break;

            default:
                // Do we have other registered types?
                $_mf = jrCore_get_registered_module_features('jrBanned', 'banned_type');
                if ($_mf && is_array($_mf)) {
                    $value = trim(strip_tags($value));
                    foreach ($_mf as $_inf) {
                        if (isset($_inf[$type]['function']) && function_exists($_inf[$type]['function'])) {
                            $func = $_inf[$type]['function'];
                            if ($tmp = $func($value, $_rt)) {
                                return $tmp;
                            }
                        }
                    }
                }
                break;
        }
    }
    return false;
}

/**
 * Check if a given type is a valid banned type
 * @param $type string
 * @return bool
 */
function jrBanned_is_valid_ban_type($type)
{
    $_tmp = jrBanned_get_banned_types();
    return (isset($_tmp[$type]));
}

/**
 * Get all banned types
 * @return array
 */
function jrBanned_get_banned_types()
{
    // Built ins
    $_opt = array(
        'ip'    => 'IP Address',
        'name'  => 'Profile or User Name',
        'email' => 'Email Address or Domain',
        'word'  => 'Forbidden Word'
    );
    $_mf  = jrCore_get_registered_module_features('jrBanned', 'banned_type');
    if ($_mf && is_array($_mf)) {
        foreach ($_mf as $_tmp) {
            foreach ($_tmp as $type => $_inf) {
                if (isset($_inf['function']) && function_exists($_inf['function'])) {
                    $_opt[$type] = $_inf['title'];
                }
            }
        }
    }
    return $_opt;
}
