<?php

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Handle file uploads via XMLHttpRequest
 */
class qqUploadedFileXhr
{
    /**
     * Save the file to the specified path
     * @param string $path
     * @param int $size size of file in bytes
     * @return boolean TRUE on success
     */
    function save($path, $size)
    {
        return jrCore_move_uploaded_media_file('php://input', $path);
    }

    function getName()
    {
        return $_REQUEST['field_name'];
    }

    function getSize()
    {
        return intval($_SERVER['CONTENT_LENGTH']);
    }
}

/**
 * Handle file uploads via regular form post (uses the $_FILES array)
 */
class qqUploadedFileForm
{
    /**
     * Save the file to the specified path
     * @param string $path
     * @return boolean TRUE on success
     */
    function save($path)
    {
        return jrCore_move_uploaded_media_file($_FILES['qqfile']['tmp_name'], $path);
    }

    function getName()
    {
        return $_FILES['qqfile']['name'];
    }

    function getSize()
    {
        return $_FILES['qqfile']['size'];
    }
}

class qqFileUploader
{
    private $allowedExtensions;
    private $sizeLimit;
    private $file;
    private $uploadName;

    function __construct(array $allowedExtensions = array(), $sizeLimit = 10485760)
    {
        $allowedExtensions = array_map("strtolower", $allowedExtensions);

        $this->allowedExtensions = $allowedExtensions;
        $this->sizeLimit         = $sizeLimit;

        if (isset($_REQUEST['field_name'])) {
            $this->file = new qqUploadedFileXhr();
        }
        elseif (isset($_FILES['qqfile'])) {
            $this->file = new qqUploadedFileForm();
        }
        else {
            $this->file = false;
        }
    }

    public function getUploadName()
    {
        if (isset($this->uploadName)) {
            return $this->uploadName;
        }
        return false;
    }

    public function getName()
    {
        if ($this->file) {
            /** @noinspection PhpUndefinedMethodInspection */
            return $this->file->getName();
        }
        return false;
    }

    /**
     * Process upload
     * @param $uploadDirectory string
     * @param bool $replaceOldFile
     * @return array
     */
    function handleUpload($uploadDirectory, $replaceOldFile = true)
    {
        global $_user, $_post;
        if (!is_writable($uploadDirectory)) {
            return array('error' => "Server error. Upload directory isn't writable.");
        }
        if (!$this->file) {
            return array('error' => 'No files were uploaded.');
        }
        /** @noinspection PhpUndefinedMethodInspection */
        $size = $this->file->getSize();
        if ($size == 0) {
            return array('error' => 'File is empty');
        }

        // Get active form session
        // we will store meta data about the file being uploaded
        $_sess = array();
        if (!empty($_post['token']) && jrCore_checktype($_post['token'], 'md5')) {
            $_sess = jrCore_form_get_session($_post['token']);
        }
        if (!jrUser_is_admin()) {
            $pid = (int) jrUser_get_profile_home_key('_profile_id');
            if ($pid != $_user['user_active_profile_id']) {
                // This user is working on a profile that is NOT their home profile
                $pid = (int) $_user['user_active_profile_id'];
                if (isset($_user['_profile_id']) && $_user['_profile_id'] == $pid) {
                    // This user's active data is already setup with active profile ID data
                    if (!empty($_user['quota_jrCore_max_upload_size'])) {
                        $max = $_user['quota_jrCore_max_upload_size'];
                    }
                }
            }
            else {
                $max = jrUser_get_profile_home_key('quota_jrCore_max_upload_size');
            }
            if (empty($max)) {
                $max = 10485760;
            }
            $this->sizeLimit = (int) $max;
            if ($size && $size > $this->sizeLimit) {
                jrCore_logger('DBG', "core: uploaded file exceeds max allowed: " . jrCore_format_size($this->sizeLimit));
                $_ln = jrUser_load_lang_strings();
                return array('error' => $_ln['jrCore'][134] . ' ' . jrCore_format_size($this->sizeLimit));
            }
        }

        $mod      = (!empty($_sess['form_params']['module'])) ? $_sess['form_params']['module'] : false;
        $nam      = $_REQUEST['field_name'];
        $ext      = jrCore_file_extension($_REQUEST[$nam]);
        $filename = $_REQUEST[$nam];
        $exts     = str_replace('.', '', $this->allowedExtensions);
        if (in_array('jpg', $exts)) {
            // Make sure we allow other JPG extensions
            foreach (array('jpeg', 'jfi', 'jfif', 'jpe') as $e) {
                if (!in_array($e, $exts)) {
                    $exts[] = $e;
                }
            }
        }
        $these = implode(', ', $exts);

        // Check for valid extension
        if ($exts && !in_array($ext, $exts)) {
            $_ln = jrUser_load_lang_strings();
            return array('error' => $_ln['jrCore'][135] . ' ' . $these);
        }

        // If multiple = false we use "0" as order id
        if (isset($_post['multiple']) && jrCore_checktype($_post['multiple'], 'is_false')) {
            $_REQUEST['orderid'] = 0;
        }

        // Trigger upload_prepare event
        $_rs = array(
            'upload_directory'         => $uploadDirectory,
            'upload_name'              => $_REQUEST['upload_name'],
            'upload_order'             => $_REQUEST['orderid'],
            'field_name'               => $nam,
            'field_allowed_extensions' => $these,
            'file_name'                => $filename,
            'file_size'                => $size,
            'file_extension'           => $ext
        );
        $_rs = jrCore_trigger_event('jrCore', 'upload_prepare', $_rs);
        if (isset($_rs['error'])) {
            // We had an error in our upload from a listener
            return array('error' => $_rs['error']);
        }

        // Copy the file to our temp directory
        // See if are doing multiple uploads for the same "field" - if we are, we need
        // to increment the field number each time
        $order = str_pad($_REQUEST['orderid'], 3, '0', STR_PAD_LEFT);
        $fname = $uploadDirectory . $order . '_' . $_REQUEST['upload_name'];
        if ($this->file->save($fname, $size)) {

            // See if this module has registered for meta data - if so, we're going
            // to automatically grab the file meta data and have it load automatically
            // when the module calls jrCore_form_get_save_data()
            if ($mod) {
                if ($_mr = jrCore_get_registered_module_features('jrCore', 'meta_data')) {
                    if (!empty($_mr[$mod]['extensions'])) {
                        if (strpos(',' . trim(trim($_mr[$mod]['extensions']), ',') . ',', $ext)) {
                            if ($_meta = jrCore_get_media_file_metadata($fname, $_REQUEST['upload_name'])) {
                                // We have our meta data from the file - save it to the active
                                // form session so it is available in the forms if needed
                                $field_name = (intval($order) > 0) ? "{$_REQUEST['upload_name']}_" . intval($order) : $_REQUEST['upload_name'];
                                $_save_data = array(
                                    '_meta' => $_meta,
                                    'order' => $order
                                );
                                jrCore_form_save_meta_data($_post['token'], $field_name, $_save_data);
                            }
                        }
                    }
                }
            }

            // Save file into to .tmp file
            jrCore_write_to_file($fname . '.tmp', $filename);

            // We have actually saved the file now - update the form session so
            // we know that files were actually uploaded during this session
            if (!empty($_post['token'])) {
                jrUser_set_session_key("form_session_uploads_{$_post['token']}", time());
            }

            // Trigger upload_saved event
            $_rs = array(
                'success'                  => true,
                'upload_directory'         => $uploadDirectory,
                'upload_name'              => $_REQUEST['upload_name'],
                'upload_order'             => $_REQUEST['orderid'],
                'field_name'               => $nam,
                'field_allowed_extensions' => $these,
                'temp_name'                => $fname,
                'file_order'               => $order,
                'file_name'                => $filename,
                'file_size'                => $size,
                'file_extension'           => $ext,
            );
            $_rs = jrCore_trigger_event('jrCore', 'upload_saved', $_rs);
            if (is_array($_rs) && isset($_rs['error'])) {
                // We had an error from a listener - cleanup
                jrCore_unlink($fname);
                jrCore_unlink($fname . '.tmp');
            }

            // We are good - trigger upload_complete
            $_rs = jrCore_trigger_event('jrCore', 'upload_complete', $_rs);

            unset($_rs['upload_directory'], $_rs['temp_name']);
            return $_rs;

        }
        return array('error' => 'Could not save uploaded file - upload was cancelled or server error encountered');
    }
}
