<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2021 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * view: system_check
 * @param array $_post Posted Data
 * @param array $_user Viewing User data
 * @param array $_conf Global Config
 */
function view_jrCore_system_check($_post, $_user, $_conf)
{
    global $_mods;
    jrUser_master_only();
    jrCore_page_include_admin_menu();
    jrCore_page_admin_tabs('jrCore');
    $refresh = jrCore_page_icon_button('refresh', 'refresh', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/system_check_update')", array('title' => 'Refresh'));
    jrCore_page_banner('system check', $refresh);
    jrCore_get_form_notice();

    $pass = jrCore_get_option_image('pass');
    $fail = jrCore_get_option_image('fail');
    $warn = jrCore_get_option_image('warning');

    $dat             = array();
    $dat[1]['title'] = 'checked';
    $dat[1]['width'] = '23%';
    $dat[2]['title'] = 'value';
    $dat[2]['width'] = '23%';
    $dat[3]['title'] = 'result';
    $dat[3]['width'] = '5%';
    $dat[4]['title'] = 'note';
    $dat[4]['width'] = '49%';
    jrCore_page_table_header($dat);

    // Get our core version from file and compare to what's in the DB
    $_mta            = jrCore_module_meta_data('jrCore');
    $dat             = array();
    $dat[1]['title'] = $_mods['jrCore']['module_name'];
    $dat[1]['class'] = 'center';
    $dat[2]['title'] = "<a href=\"" . jrCore_get_base_url() . "/{$_post['module_url']}/admin/info\"><b><u>{$_mta['version']}</u></b></a>";
    $dat[2]['class'] = 'center';
    $dat[3]['title'] = $pass;
    $dat[3]['class'] = 'center';
    $dat[4]['title'] = 'Jamroom Core v' . $_mods['jrCore']['module_version'];
    if ($_mta['version'] != $_mods['jrCore']['module_version']) {
        $dat[3]['title'] = $fail;
        $dat[4]['title'] .= '&nbsp;&nbsp;<a href="' . jrCore_get_base_url() . '/' . jrCore_get_module_url('jrCore') . '/integrity_check"><strong><u>Integrity Check Required!</u></strong>';
    }
    jrCore_page_table_row($dat);

    // Server
    $dat             = array();
    $dat[1]['title'] = 'Server OS';
    $dat[1]['class'] = 'center';
    $dat[2]['title'] = jrCore_get_server_os();
    $dat[2]['class'] = 'center';
    $dat[3]['title'] = $pass;
    $dat[3]['class'] = 'center';
    $dat[4]['title'] = 'Linux OS required';
    jrCore_page_table_row($dat);

    $web_server = php_sapi_name();
    if (strpos(' ' . $web_server, 'apache2handler')) {
        $web_server = 'Apache';
        // Get version
        if (isset($_SERVER['SERVER_SOFTWARE']) && strpos($_SERVER['SERVER_SOFTWARE'], '/')) {
            list(, $apache_version) = explode('/', $_SERVER['SERVER_SOFTWARE']);
            $apache_version = jrCore_string_field($apache_version, 1);
            if (strpos($apache_version, '.')) {
                $web_server .= ' ' . $apache_version;
            }
        }
    }

    // Web Server
    $dat             = array();
    $dat[1]['title'] = 'Web Server';
    $dat[1]['class'] = 'center';
    $dat[2]['title'] = $web_server;
    $dat[2]['class'] = 'center word-break';
    $dat[3]['title'] = $pass;
    $dat[3]['class'] = 'center';
    $dat[4]['title'] = 'Apache or Nginx required, Apache recommended';
    jrCore_page_table_row($dat);

    // PHP Version
    list($php_version,) = explode('+', phpversion());
    $result = $fail;
    if (version_compare($php_version, '7.2.0') != -1) {
        $result = $pass;
    }

    $dat             = array();
    $dat[1]['title'] = 'PHP Version';
    $dat[1]['class'] = 'center';
    $dat[2]['title'] = "<a onclick=\"window.open('" . jrCore_get_base_url() . '/' . $_post['module_url'] . "/phpinfo')\"><b><u>{$php_version}</u></b></a>";
    $dat[2]['class'] = 'center';
    $dat[3]['title'] = $result;
    $dat[3]['class'] = 'center';
    $dat[4]['title'] = 'PHP 7.2+ required';
    jrCore_page_table_row($dat);

    // DB Version
    $_db = jrCore_db_query("SHOW VARIABLES WHERE Variable_name = 'version'", 'SINGLE');
    if ($_db && is_array($_db) && isset($_db['Value'])) {
        $ver = $_db['Value'];
    }
    else {
        $msi = jrCore_db_connect();
        $ver = mysqli_get_server_info($msi);
    }
    $ven = '';
    if (strpos($ver, '-')) {
        list($ver, $ven) = explode('-', $ver);
        if (stripos(' ' . $ven, 'maria')) {
            $ven = 'MariaDB';
        }
        elseif (stripos(' ' . $ven, 'mysql')) {
            $ven = 'MySQL';
        }
    }
    $result = $pass;
    if (strpos($ver, '3.') === 0 || strpos($ver, '4.') === 0 || strpos($ver, '5.0') === 0 || strpos($ver, '5.1') === 0 || strpos($ver, '5.5') === 0) {
        $result = $fail;
    }

    $dat             = array();
    $dat[1]['title'] = 'Database';
    $dat[1]['class'] = 'center';
    $dat[2]['title'] = "<a onclick=\"window.open('" . jrCore_get_base_url() . '/' . $_post['module_url'] . "/database_info')\"><b><u>" . $ven . ' ' . $ver . "</u></b></a>";
    $dat[2]['class'] = 'center';
    $dat[3]['title'] = $result;
    $dat[3]['class'] = 'center';
    $dat[4]['title'] = 'MySQL 5.7.0 minimum, MySQL 8+ or MariaDB 10.1+ recommended';
    jrCore_page_table_row($dat);

    // Disabled Functions
    $dis_funcs = ini_get('disable_functions');
    if ($dis_funcs && $dis_funcs != '') {
        $dis_funcs = explode(',', $dis_funcs);
        if (isset($dis_funcs) && is_array($dis_funcs)) {
            foreach ($dis_funcs as $k => $fnc) {
                // We don't care about disabled process control functions
                $fnc = trim($fnc);
                if (strlen($fnc) === 0 || strpos($fnc, 'pcntl') === 0) {
                    unset($dis_funcs[$k]);
                }
                // Other functions we do not care about as Jamroom does not use them
                switch ($fnc) {
                    case 'dl':
                        unset($dis_funcs[$k]);
                        break;
                }
            }
        }
        if (isset($dis_funcs) && count($dis_funcs) > 0) {
            $dis_funcs = implode('<br>', $dis_funcs);
            $result    = $fail;

            $dat             = array();
            $dat[1]['title'] = 'Disabled Functions';
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = $dis_funcs;
            $dat[2]['class'] = 'center';
            $dat[3]['title'] = $result;
            $dat[3]['class'] = 'center';
            $dat[4]['title'] = 'Disabled PHP Functions can impact system functionality';
            jrCore_page_table_row($dat);
        }
    }

    // Max Allowed Packet
    $req = "SHOW VARIABLES LIKE 'max_allowed_packet'";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if ($_rt && isset($_rt['Value'])) {
        $map = round(($_rt['Value'] / 1048576), 1);
        if ($map < 4) {
            $dat             = array();
            $dat[1]['title'] = 'Max Query Size';
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = $map . 'mb';
            $dat[2]['class'] = 'center';
            $dat[3]['title'] = $warn;
            $dat[4]['title'] = 'Recommended: increase max_allowed_packet to 4MB or higher:<br><a href="http://dev.mysql.com/doc/refman/5.1/en/packet-too-large.html" target="_blank" rel="noreferrer noopener"><u>Increasing Max Allowed Packet</u></a>';
            $dat[3]['class'] = 'center';
            jrCore_page_table_row($dat);
        }
    }

    // Local Cache
    if (!jrCore_local_cache_is_enabled()) {
        $dat             = array();
        $dat[1]['title'] = 'Memory Cache';
        $dat[1]['class'] = 'center';
        $dat[2]['title'] = 'functions are missing';
        $dat[2]['class'] = 'center';
        $dat[3]['title'] = $fail;
        $dat[4]['title'] = 'Ensure <a href="http://us2.php.net/apcu" target="_blank" rel="noreferrer noopener"><u>APCu functions</u></a> are enabled in your PHP install';
        $dat[3]['class'] = 'center';
        jrCore_page_table_row($dat);
    }

    // Directories
    $_to_check = array('cache', 'logs', 'media');
    $_bad      = array();
    foreach ($_to_check as $dir) {
        if (!is_dir(APP_DIR . "/data/{$dir}")) {
            // See if we can create it
            if (!jrCore_create_directory(APP_DIR . "/data/{$dir}")) {
                $_bad[] = "data/{$dir} does not exist";
            }
        }
        elseif (!is_writable(APP_DIR . "/data/{$dir}")) {
            chmod(APP_DIR . "/data/{$dir}", $_conf['jrCore_dir_perms']);
            if (!is_writable(APP_DIR . "/data/{$dir}")) {
                $_bad[] = "data/{$dir} is not writable";
            }
        }
    }
    if (isset($_bad) && is_array($_bad) && count($_bad) > 0) {
        $note   = 'All directories <strong>must be writable</strong> by web user!';
        $dirs   = implode('<br>', $_bad);
        $result = $fail;
    }
    else {
        $note   = 'All directories are writable';
        $dirs   = 'all writable';
        $result = $pass;
    }
    $dat             = array();
    $dat[1]['title'] = 'Data Directories';
    $dat[1]['class'] = 'center';
    $dat[2]['title'] = $dirs;
    $dat[2]['class'] = 'center';
    $dat[3]['title'] = $result;
    $dat[3]['class'] = 'center';
    $dat[4]['title'] = $note;
    jrCore_page_table_row($dat);

    $upl             = jrCore_get_max_allowed_upload();
    $dat             = array();
    $dat[1]['title'] = 'Max Upload';
    $dat[1]['class'] = 'center';
    $dat[2]['title'] = jrCore_format_size($upl);
    $dat[2]['class'] = 'center';
    $dat[3]['title'] = ($upl <= 2097152) ? $fail : $pass;
    $dat[3]['class'] = 'center';
    $dat[4]['title'] = (($upl <= 2097152) ? 'Increase post_max_size and upload_max_filesize in your php.ini to allow larger uploads<br>' : '');
    $dat[4]['title'] .= 'View the <a href="https://www.jamroom.net/the-jamroom-network/documentation/problems/748/how-do-i-increase-phps-upload-limit" target="_blank" rel="noreferrer noopener"><u>FAQ on increasing the allowed upload size</u></a>';
    jrCore_page_table_row($dat);

    // Check installed locale's
    if (isset($_conf['jrUser_default_language']) && $_conf['jrUser_default_language'] != 'en-US') {
        if ($_lc = jrCore_get_installed_locales()) {
            $dat             = array();
            $dat[1]['title'] = 'Supported Languages';
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = count($_lc);
            $dat[2]['class'] = 'center';

            // Is the default language supported?
            if (!isset($_lc["{$_conf['jrUser_default_language']}"])) {
                $dat[3]['title'] = $fail;
                $dat[4]['title'] = "Language support (locale) for {$_conf['jrUser_default_language']} is not installed on this server";
            }
            else {
                $dat[3]['title'] = $pass;
                $dat[4]['title'] = "Language support (locale) for {$_conf['jrUser_default_language']} is installed and active";
            }
            $dat[3]['class'] = 'center';
            jrCore_page_table_row($dat);
        }
    }

    // Apache rlimits
    if (function_exists('posix_getrlimit')) {
        $_rl = posix_getrlimit();

        // Apache RlimitMEM
        if ((jrCore_checktype($_rl['soft totalmem'], 'number_nz') && $_rl['soft totalmem'] < 67108864) || (jrCore_checktype($_rl['hard totalmem'], 'number_nz') && $_rl['hard totalmem'] < 67108864)) {
            $apmem = $_rl['soft totalmem'];
            if (jrCore_checktype($_rl['hard totalmem'], 'number_nz') && $_rl['hard totalmem'] < $_rl['soft totalmem']) {
                $apmem = $_rl['hard totalmem'];
            }
            $show            = (($apmem / 1024) / 1024);
            $dat             = array();
            $dat[1]['title'] = 'Apache Memory Limit';
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = $show . 'MB';
            $dat[2]['class'] = 'center';
            $dat[3]['title'] = $fail;
            $dat[3]['class'] = 'center';
            $dat[4]['title'] = "Apache is limiting the amount of memory you can use - this could cause issues, especially when doing Media Conversions. Apache Memory Limits are put in place by your hosting provider, and cannot be modified - contact your hosting provider and have them increase the limit, or set it to &quot;unlimited&quot;.";
            jrCore_page_table_row($dat);
        }
        // Apache RlimitCPU
        if (jrCore_checktype($_rl['soft cpu'], 'number_nz') && $_rl['soft cpu'] < 20) {
            $dat             = array();
            $dat[1]['title'] = 'Apache Soft CPU Limit';
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = $_rl['soft cpu'];
            $dat[2]['class'] = 'center';
            $dat[3]['title'] = $fail;
            $dat[3]['class'] = 'center';
            $dat[4]['title'] = "Apache is limiting the amount of CPU you can use - this could cause issues, especially when doing Media Conversions. Apache CPU Limits are put in place by your hosting provider, and cannot be modified - you will want to contact your hosting provider and have them set the soft cpu limit to &quot;unlimited&quot;.";
            jrCore_page_table_row($dat);
        }
        elseif (jrCore_checktype($_rl['hard cpu'], 'number_nz') && $_rl['hard cpu'] < 40) {
            $dat             = array();
            $dat[1]['title'] = 'Apache Hard CPU Limit';
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = $_rl['hard cpu'];
            $dat[2]['class'] = 'center';
            $dat[3]['title'] = $fail;
            $dat[3]['class'] = 'center';
            $dat[4]['title'] = "Apache is limiting the amount of CPU you can use - this could cause issues, especially when doing Media Conversions. Apache CPU Limits are put in place by your hosting provider, and cannot be modified - you will want to contact your hosting provider and have them set the soft cpu limit to &quot;unlimited&quot;.";
            jrCore_page_table_row($dat);
        }

        // Apache RlimitNPROC
        if ((jrCore_checktype($_rl['soft maxproc'], 'number_nz') && $_rl['soft maxproc'] < 200) || (jrCore_checktype($_rl['hard maxproc'], 'number_nz') && $_rl['hard maxproc'] < 200)) {
            $approc = $_rl['soft maxproc'];
            if (jrCore_checktype($_rl['hard maxproc'], 'number_nz') && $_rl['hard maxproc'] < $_rl['soft maxproc']) {
                $approc = $_rl['hard maxproc'];
            }
            $dat             = array();
            $dat[1]['title'] = 'Apache Process Limit';
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = $approc;
            $dat[2]['class'] = 'center';
            $dat[3]['title'] = $fail;
            $dat[3]['class'] = 'center';
            $dat[4]['title'] = "Apache is limiting the amount of Processes you can use - this could cause issues, especially when doing Media Conversions. Apache PROC Limits are put in place by your hosting provider, and cannot be modified - you will want to contact your hosting provider and have them set the soft and hard maxproc limits to &quot;unlimited&quot;.";
            jrCore_page_table_row($dat);
        }
    }
    $dat             = array();
    $dat[1]['title'] = 'module';
    $dat[2]['title'] = 'checked';
    $dat[3]['title'] = 'result';
    $dat[4]['title'] = 'note';
    jrCore_page_table_header($dat, null, true);

    // Go through installed modules
    foreach ($_mods as $mod => $_inf) {
        if ($mod == 'jrCore' || $mod == 'jrSystemTools' || !jrCore_module_is_active($mod)) {
            continue;
        }
        // Check if this module requires other modules to function - make sure they exist and are activated
        if (!empty($_inf['module_requires'])) {
            $_req = explode(',', $_inf['module_requires']);
            if (is_array($_req)) {
                foreach ($_req as $rmod) {
                    // See if we have been given an explicit version - i.e. jrImage:1.1.5
                    if (strpos($rmod, ':')) {
                        list($rmod, $vers) = explode(':', $rmod);
                        $rmod = trim($rmod);
                        $vers = trim($vers);
                    }
                    else {
                        $rmod = trim($rmod);
                        $vers = '0.0.0';
                    }
                    if (!is_dir(APP_DIR . "/modules/{$rmod}")) {
                        $murl            = jrCore_get_module_url('jrMarket');
                        $dat             = array();
                        $dat[1]['title'] = $_inf['module_name'];
                        $dat[1]['class'] = 'center';
                        $dat[2]['title'] = 'required module: ' . $rmod;
                        $dat[2]['class'] = 'center';
                        $dat[3]['title'] = $fail;
                        $dat[3]['class'] = 'center';
                        $dat[4]['title'] = "<strong>{$rmod}</strong> module is missing. <a href=\"" . jrCore_get_base_url() . "/{$murl}/browse/module?search_string={$rmod}\" style=\"text-decoration:underline\" target=\"_blank\">Search in Marketplace</a>";
                        jrCore_page_table_row($dat);
                    }
                    elseif (!jrCore_module_is_active($rmod)) {
                        $murl            = jrCore_get_module_url($rmod);
                        $dat             = array();
                        $dat[1]['title'] = $_inf['module_name'];
                        $dat[1]['class'] = 'center';
                        $dat[2]['title'] = 'required module: ' . $rmod;
                        $dat[2]['class'] = 'center';
                        $dat[3]['title'] = $fail;
                        $dat[3]['class'] = 'center';
                        $dat[4]['title'] = "<strong>{$rmod}</strong> module is not active, <a href=\"" . jrCore_get_base_url() . "/{$murl}/admin/info/hl=module_active\" style=\"text-decoration:underline\" target=\"_blank\">click here</a>";
                        jrCore_page_table_row($dat);
                    }
                    elseif (version_compare($_mods[$rmod]['module_version'], $vers, '<')) {
                        $dat             = array();
                        $dat[1]['title'] = $_inf['module_name'];
                        $dat[1]['class'] = 'center';
                        if ($vers != '0.0.0') {
                            $dat[2]['title'] = 'required module: ' . $rmod . ' ' . $vers;
                        }
                        else {
                            $dat[2]['title'] = 'required module: ' . $rmod;
                        }
                        $dat[2]['class'] = 'center';
                        $dat[3]['title'] = $fail;
                        $dat[3]['class'] = 'center';
                        $dat[4]['title'] = "<strong>{$rmod}</strong> version {$vers} required (current: {$_mods["{$rmod}"]['module_version']})";
                        jrCore_page_table_row($dat);
                    }
                }
            }
        }
        // See if this module has any additional checks to add
        $_inf['pass']    = $pass;
        $_inf['fail']    = $fail;
        $_inf['warning'] = jrCore_get_option_image('warning');
        jrCore_trigger_event('jrCore', 'system_check', array(), $_inf, $mod);
    }

    // System Tools
    if (jrCore_module_is_active('jrSystemTools')) {
        $dat             = array();
        $dat[1]['title'] = 'system tools';
        $dat[2]['title'] = 'value';
        $dat[3]['title'] = 'result';
        $dat[4]['title'] = 'note';
        jrCore_page_table_header($dat, null, true);
        $_inf['pass']    = $pass;
        $_inf['fail']    = $fail;
        $_inf['warning'] = jrCore_get_option_image('warning');
        jrCore_trigger_event('jrCore', 'system_check', array(), $_inf, 'jrSystemTools');
    }

    jrCore_page_table_footer();
    jrCore_page_cancel_button("{$_conf['jrCore_base_url']}/{$_post['module_url']}/admin/tools");
    jrCore_page_display();
}
