<?php
 /**
 * Jamroom Simple Custom Forms module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrCustomForm_meta()
{
    return array(
        'name'        => 'Simple Custom Forms',
        'url'         => 'form',
        'version'     => '1.3.8',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Create simple forms that store responses and optionally email admin users',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/2885/simple-custom-forms',
        'category'    => 'forms',
        'requires'    => 'jrCore:6.5.12',
        'license'     => 'jcl'
    );
}

/**
 * init
 */
function jrCustomForm_init()
{
    // Register our custom CSS
    jrCore_register_module_feature('jrCore', 'css', 'jrCustomForm', 'jrCustomForm.css');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrCustomForm', true);

    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCustomForm', 'browse', array('Form Browser', 'Browse existing Custom Forms'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrCustomForm', 'create', array('Create a Custom Form', 'Create a new Custom Form'));

    // Custom tabs
    jrCore_register_module_feature('jrCore', 'admin_tab', 'jrCustomForm', 'browse', 'Form Browser');

    // Our default master view
    jrCore_register_module_feature('jrCore', 'default_admin_view', 'jrCustomForm', 'browse');

    // Maintain proper form response counts
    jrCore_register_event_listener('jrCore', 'verify_module', 'jrCustomForm_verify_module_listener');
    jrCore_register_event_listener('jrCore', 'db_delete_item', 'jrCustomForm_db_delete_item_listener');
    jrCore_register_event_listener('jrCore', 'form_validate_init', 'jrCustomForm_form_validate_init_listener');

    // notifications
    $_tmp = array(
        'label' => 'new form response',
        'help'  => 'When a new form response has been received, how do you want to be notified?',
        'group' => 'admin'
    );
    jrCore_register_module_feature('jrUser', 'notification', 'jrCustomForm', 'form_response', $_tmp);

    // Site Builder widget
    jrCore_register_module_feature('jrSiteBuilder', 'widget', 'jrCustomForm', 'widget_form', 'Embedded Form');

    // Akismet Anti Spam support
    jrCore_register_module_feature('jrAkismet', 'spam_check', 'jrCustomForm', 'contact-form', 'reject');

    return true;
}

//------------------------------------
// EVENT LISTENERS
//------------------------------------

/**
 * Make sure we're not using an internal form name
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrCustomForm_form_validate_init_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (isset($_post['_uri']) && strpos($_post['_uri'], '/form_designer_save/') && isset($_post['new_name'])) {
        switch ($_post['new_name']) {
            case 'form_id':
            case 'form_created':
            case 'form_updated':
            case 'form_name':
            case 'form_title':
            case 'form_message':
            case 'form_unique':
            case 'form_login':
            case 'form_notify':
            case 'form_responses':
                jrCore_set_form_notice('error', "<b>{$_post['new_name']}</b> is used internally - please use a different New Field Name", false);
                jrCore_form_result();
                break;
        }
    }
    return $_data;
}

/**
 * Maintain form response counts
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrCustomForm_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // get all forms
    $tbl = jrCore_db_table_name('jrCustomForm', 'form');
    $req = "SELECT form_id, form_name FROM {$tbl}";
    $_rt = jrCore_db_query($req, 'form_id', false, 'form_name');
    if ($_rt && is_array($_rt)) {
        foreach ($_rt as $fid => $fname) {
            $res = (int) jrCore_db_run_key_function('jrCustomForm', 'form_name', $fname, 'count');
            if (!$res || !jrCore_checktype($res, 'number_nz')) {
                $res = 0;
            }
            $req = "UPDATE {$tbl} SET form_responses = {$res} WHERE form_id = " . intval($fid);
            jrCore_db_query($req);
        }
    }

    // Some fields are no longer needed...
    jrCore_db_delete_key_from_all_items('jrCustomForm', 'form_created');
    jrCore_db_delete_key_from_all_items('jrCustomForm', 'form_user_id');
    jrCore_db_delete_key_from_all_items('jrCustomForm', 'form_user_name');
    jrCore_db_delete_key_from_all_items('jrCustomForm', 'form_profile_name');

    return $_data;
}

/**
 * Decrement response counts
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrCustomForm_db_delete_item_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_args['module']) && $_args['module'] == 'jrCustomForm' && isset($_data['form_name']) && strlen($_data['form_name']) > 0) {
        // Decrement counts
        $tbl = jrCore_db_table_name('jrCustomForm', 'form');
        $req = "UPDATE {$tbl} SET form_responses = (form_responses - 1) WHERE form_name = '" . jrCore_db_escape($_data['form_name']) . "' AND form_responses > 0";
        $cnt = jrCore_db_query($req, 'COUNT');
        if ($cnt === 0) {
            jrCore_logger('MAJ', "form: unable to decrement form_responses count for form: {$_data['form_name']}");
        }
    }
    return $_data;
}

//------------------------------------
// WIDGETS
//------------------------------------

/**
 * Display CONFIG screen for Embedded Form
 * @param $_post array Post info
 * @param $_user array User array
 * @param $_conf array Global Config
 * @param $_wg array Widget info
 * @return bool
 */
function jrCustomForm_widget_form_config($_post, $_user, $_conf, $_wg)
{
    // Get available forms
    $tbl  = jrCore_db_table_name('jrCustomForm', 'form');
    $req  = "SELECT form_name, form_title FROM {$tbl} ORDER BY form_title ASC";
    $_opt = jrCore_db_query($req, 'form_name', false, 'form_title');

    // Embed form
    $_tmp = array(
        'name'     => 'name',
        'label'    => 'Form to Embed',
        'help'     => 'Select the Form you would like to embed in this container location',
        'options'  => $_opt,
        'default'  => '',
        'type'     => 'select',
        'validate' => 'printable',
        'size'     => 8
    );
    jrCore_form_field_create($_tmp);

    // Show Help Buttons
    $_tmp = array(
        'name'     => 'help',
        'label'    => 'Show Form Help',
        'help'     => 'If this option is checked and a form field has help text, the help button will show to the right of the form field.',
        'default'  => 'off',
        'type'     => 'checkbox',
        'validate' => 'onoff'
    );
    jrCore_form_field_create($_tmp);
    return true;
}

/**
 * Get Widget results from posted Config data
 * @param $_post array Post info
 * @return array
 */
function jrCustomForm_widget_form_config_save($_post)
{
    return array(
        'name' => $_post['name'],
        'help' => $_post['help']
    );
}

/**
 * HTML Editor Widget DISPLAY
 * @param $_widget array Page Widget info
 * @return string
 */
function jrCustomForm_widget_form_display($_widget)
{
    return jrCore_parse_template('widget_embed_form.tpl', array('params' => $_widget), 'jrCustomForm');
}

/**
 * Embed a Custom Form into a template
 * @param $params array Function params
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrCustomForm_embed_form($params, $smarty)
{
    global $_conf;

    // Backwards compatible
    if (empty($params['params'])) {
        if (!empty($params['name'])) {
            $help   = ($params['help']) ? $params['help'] : 'off';
            $params = array(
                'params' => array(
                    'name' => $params['name'],
                    'help' => $help
                )
            );
        }
        else {
            return jrCore_smarty_missing_error('params');
        }
    }

    $tbl = jrCore_db_table_name('jrCustomForm', 'form');
    $req = "SELECT * FROM {$tbl} WHERE `form_name` = '" . jrCore_db_escape($params['params']['name']) . "' LIMIT 1";
    $_fm = jrCore_db_query($req, 'SINGLE');
    if (!$_fm || !is_array($_fm)) {
        return jrCore_smarty_invalid_error('name');
    }
    if (isset($_fm['form_login']) && $_fm['form_login'] == 'on' && !jrUser_is_logged_in()) {
        $out = jrCore_parse_template('form_embed_login_notice.tpl', $_fm, 'jrCustomForm');
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], $out);
            return '';
        }
        return $out;
    }

    $_ln = jrUser_load_lang_strings();

    // Form init (establish form session)
    $url = jrCore_get_module_url('jrCustomForm');
    $act = "{$_conf['jrCore_base_url']}/{$url}/{$params['params']['name']}_save";
    $_fr = array(
        'name'        => $_fm['form_name'],
        'module'      => 'jrCustomForm',
        'action'      => $act,
        'success_msg' => $_ln['jrCustomForm'][2]
    );
    $tkn = jrCore_form_begin($_fm['form_name'], $act, $_fr);
    jrCore_form_create_session($tkn, $_fr);

    // Get fields
    $_fd = jrCore_get_designer_form_fields('jrCustomForm', $params['params']['name']);
    if ($_fd && is_array($_fd)) {
        jrCore_set_flag('jrcore_is_mobile_device', 'yes');
        $idx = 2;
        foreach ($_fd as $_field) {
            if (!isset($_field['active']) || $_field['active'] != 1) {
                // Not showing this one
                continue;
            }
            if (!empty($params['params']['help']) && $params['params']['help'] == 'off') {
                // We are not showing help - remove button and set width
                unset($_field['help']);
                $_field['class'] = 'custom-form-embed-width';
            }
            $_field['form_designer'] = false;
            jrCore_form_field_create($_field, 'jrCustomForm');
            $idx++;
        }
        // If the user is NOT logged on, make sure they are human
        if (!jrUser_is_logged_in() && !jrCore_get_flag('jrcore_form_field_checkbox_spambot')) {
            $_tmp = array(
                'name'          => 'form_is_human',
                'label'         => $_ln['jrUser'][90],
                'help'          => $_ln['jrUser'][91],
                'type'          => 'checkbox_spambot',
                'error_msg'     => $_ln['jrUser'][92],
                'validate'      => 'onoff',
                'form_designer' => false
            );
            jrCore_form_field_create($_tmp);
            $sbid = jrCore_get_flag('jrcore_form_field_checkbox_spambot');
            $html = "<script type=\"text/javascript\">$(document).ready(function(){ jrFormSpamBotCheckbox('{$sbid}',{$idx}) });</script>";
            jrCore_page_custom($html);
        }
    }
    jrCore_page_set_no_header_or_footer();
    $out = jrCore_page_display(true);
    jrCore_delete_flag('jrcore_is_mobile_device');
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

//------------------------------------
// FUNCTIONS
//------------------------------------

/**
 * Get an email address from a form response
 * @param $_it array Response
 * @return bool|mixed
 */
function jrCustomForm_get_email_from_response($_it)
{
    if (!is_array($_it)) {
        return false;
    }
    foreach ($_it as $k => $v) {
        if (strpos($k, 'email') && jrCore_checktype($v, 'email')) {
            return $v;
        }
    }
    return false;
}

/**
 * Format a response item
 * @param $_it array Item from the form DS
 * @return bool|string
 */
function jrCustomForm_format_response($_it)
{
    global $_conf;
    if (!is_array($_it)) {
        return false;
    }
    ksort($_it);
    $url = jrCore_get_module_url('jrCustomForm');
    $crl = jrCore_get_module_url('jrUser');
    $_tm = array();
    $_rp = array("\n", "\r", "\n\r");
    $_fl = false;

    // First - process for any FILES
    foreach ($_it as $k => $v) {
        if (strpos($k, '_extension')) {
            if (!is_array($_fl)) {
                $_fl = array();
            }
            $fld       = trim(str_replace('_extension', '', $k));
            $_fl[$fld] = $fld;
        }
    }
    foreach ($_it as $k => $v) {
        if ($k == '_created') {
            $_tm[] = "<span class=\"ds_browser_key form_browser_key\">received:</span> <span class=\"ds_browser_value form_browser_value\">" . jrCore_format_time($v) . "</span>";
            continue;
        }
        elseif ($k == 'form_user_ip') {
            $_tm[] = "<span class=\"ds_browser_key form_browser_key\">user_ip:</span> <span class=\"ds_browser_value form_browser_value\"><a onclick=\"popwin('{$_conf['jrCore_base_url']}/{$crl}/whois/{$v}','{$v}',900,600,'yes');\"><u>{$v}</u></a></span>";
            continue;
        }
        if (strpos($k, 'form_') !== 0) {
            continue;
        }
        switch ($k) {
            case 'form_name':
                continue 2;
        }
        if ($_fl) {
            foreach ($_fl as $fld) {
                if (strpos($k, "{$fld}_") === 0) {
                    if (!strpos($k, '_name')) {
                        continue 2;
                    }
                    $v = "<a href=\"{$_conf['jrCore_base_url']}/{$url}/download/{$fld}/{$_it['_item_id']}\"><u>{$v}</u></a> (" . jrCore_format_size($_it["{$fld}_size"]) . ')';
                }
            }
        }
        $st = true;
        if (strpos($v, '<a href') === 0) {
            $st = false;
        }
        elseif (isset($v) && is_array($v)) {
            $v  = json_encode($v);
            $st = false;
        }
        elseif (is_numeric($v) && strlen($v) === 10) {
            $v  = jrCore_format_time($v);
            $st = false;
        }
        if ($st) {
            $v = strip_tags(str_replace($_rp, ' ', $v));
        }
        $_tm[] = "<span class=\"ds_browser_key form_browser_key\">" . substr($k, 5) . ":</span> <span class=\"ds_browser_value form_browser_value\">{$v}</span>";
    }
    return implode('<br>', $_tm);
}

