<?php
 /**
 * Jamroom Geo Location module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

// MaxMind v2 reader
use GeoIp2\Database\Reader;
use GeoIp2\WebService\Client;

/**
 * meta
 */
function jrGeo_meta()
{
    return array(
        'name'        => 'Geo Location',
        'url'         => 'geo',
        'version'     => '2.3.1',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Adds Geo Location tools and functionality to the system',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/863/geo-location',
        'license'     => 'mpl',
        'php_minv'    => '7.2.0',
        'category'    => 'tools'
    );
}

/**
 * init
 */
function jrGeo_init()
{
    jrCore_register_module_feature('jrCore', 'css', 'jrGeo', 'jrGeo.css');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrGeo', 'jrGeo.js');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrGeo', 'jrGeo_admin.js', 'admin');

    jrCore_register_module_feature('jrCore', 'tool_view', 'jrGeo', 'database', array('Update Geo Databases', 'Install new or updated Geo Location Database Files'));
    jrCore_register_module_feature('jrCore', 'admin_tab', 'jrGeo', 'database', 'Geo Databases');
    jrCore_register_module_feature('jrCore', 'default_admin_view', 'jrGeo', 'database');

    // Watch for special "zip_code" searches
    jrCore_register_event_listener('jrCore', 'db_search_params', 'jrGeo_db_search_params_listener');

    // Make sure we have the city dat file uploaded
    jrCore_register_event_listener('jrCore', 'system_check', 'jrGeo_system_check_listener');

    // Verify
    jrCore_register_event_listener('jrCore', 'verify_module', 'jrGeo_verify_module_listener');

    // Signup validate
    jrCore_register_event_listener('jrUser', 'signup_validate', 'jrGeo_signup_validate_listener');

    // Site Builder widget
    jrCore_register_module_feature('jrSiteBuilder', 'widget', 'jrGeo', 'zip_search', 'ZIP Code Search');

    // Geocode worker
    jrCore_register_queue_worker('jrGeo', 'geocode_address', 'jrGeo_geocode_address_worker', 0, 2, 120, NORMAL_PRIORITY_QUEUE);
    jrCore_register_queue_worker('jrGeo', 'install_zipcode_db', 'jrGeo_install_zipcode_db_worker', 0, 1, 7200, NORMAL_PRIORITY_QUEUE);

    // update the _geocode_lat and _geocode_lng when _zip changes
    jrCore_register_event_listener('jrCore', 'db_update_item', 'jrGeo_db_update_item_listener');

    // Banned bots
    $_tmp = array(
        'title'    => 'Signup Country',
        'help'     => 'You can ban signups from specific countries baed on the user IP Address.  Enter a name here that if found anywhere in the country_name IP address field will present the user with an &quot;account already exists&quot; error message',
        'function' => 'jrGeo_is_banned_country'
    );
    jrCore_register_module_feature('jrBanned', 'banned_type', 'jrGeo', 'banned_country', $_tmp);

    return true;
}

//-------------------
// Queue Worker
//-------------------

/**
 * Get lat and lng for a physical address and update the datastore
 * @param array $_queue The queue entry the worker will receive
 * @return bool
 */
function jrGeo_geocode_address_worker($_queue)
{
    $pfx = jrCore_db_get_prefix($_queue['module']);
    if (isset($_queue['_item']["{$pfx}_geocode_lat"]) && isset($_queue['_item']["{$pfx}_geocode_lng"])) {
        $_up = array(
            "{$pfx}_geocode_lat" => $_queue['_item']["{$pfx}_geocode_lat"],
            "{$pfx}_geocode_lng" => $_queue['_item']["{$pfx}_geocode_lng"]
        );
        jrCore_db_update_item($_queue['module'], $_queue['_item']['_item_id'], $_up, null, false);
    }
    elseif ($address = jrGeo_get_address_from_item_data($_queue['module'], $_queue['_item'])) {
        if ($_gc = jrGeo_get_geocode_for_address($address)) {
            $_up = array(
                "{$pfx}_geocode_lat" => $_gc['latitude'],
                "{$pfx}_geocode_lng" => $_gc['longitude']
            );
            jrCore_db_update_item($_queue['module'], $_queue['_item']['_item_id'], $_up, null, false);
        }
    }
    return true;
}

/**
 * Install the ZIP Code database
 * @param array $_queue The queue entry the worker will receive
 * @return bool
 */
function jrGeo_install_zipcode_db_worker($_queue)
{
    if (isset($_queue['file']) && is_file($_queue['file'])) {
        jrGeo_load_zip_database($_queue['file']);
    }
    return true;
}

//-----------------------------
// Widget
//-----------------------------

/**
 * Display CONFIG screen for Widget
 * @param $_post array Post info
 * @param $_user array User array
 * @param $_conf array Global Config
 * @param $_wg array Widget info
 * @return bool
 */
function jrGeo_zip_search_config($_post, $_user, $_conf, $_wg)
{
    global $_mods;

    // module
    $_opt = jrCore_get_datastore_modules();
    foreach ($_opt as $mod => $url) {
        if (!jrCore_module_is_active($mod)) {
            unset($_opt[$mod]);
            continue;
        }
        if (is_file(APP_DIR . "/modules/{$mod}/templates/index.tpl")) {
            $_opt[$mod] = $_mods[$mod]['module_name'];
        }
        else {
            unset($_opt[$mod]);
        }
    }
    $_opt = array_merge(array('_' => '- Select a Module -'), $_opt);

    $mod = '_';
    if (!empty($_wg['widget_data']['zip_module'])) {
        $mod = $_wg['widget_data']['zip_module'];
    }

    $_tmp = array(
        'name'     => 'zip_module',
        'label'    => 'Search Module',
        'help'     => 'Select the module that contains the items you want to search',
        'options'  => $_opt,
        'value'    => $mod,
        'default'  => $mod,
        'type'     => 'select',
        'onchange' => 'jrGeo_zip_search_get_module_keys(this)',
        'validate' => 'printable'
    );
    jrCore_form_field_create($_tmp);

    $_tmp = array(
        'name'     => 'zip_key',
        'label'    => 'Search Key',
        'help'     => 'Select the datastore key that contains the ZIP Code you want to search',
        'value'    => (isset($_wg['widget_data']['zip_key'])) ? $_wg['widget_data']['zip_key'] : '_',
        'type'     => 'select',
        'options'  => ($mod != '_') ? jrGeo_get_zip_keys_for_module($mod) : array(),
        'validate' => false,
        'disabled' => 'disabled',
        'class'    => 'form_element_disabled'
    );
    jrCore_form_field_create($_tmp);
    return true;
}

/**
 * Get Widget results from posted Config data
 * @param $_post array Post info
 * @return array
 */
function jrGeo_zip_search_config_save($_post)
{
    $_data = array(
        'zip_module' => $_post['zip_module'],
        'zip_key'    => $_post['zip_key'],
    );
    return array('widget_data' => $_data);
}

/**
 * Widget DISPLAY
 * @param $_widget array Page Widget info
 * @return string
 */
function jrGeo_zip_search_display($_widget)
{
    return jrCore_parse_template('zip_search_form.tpl', $_widget, 'jrGeo');
}

//-----------------------------
// Event Listeners
//-----------------------------

/**
 * Watch for Zip code search parameters in jrCore_db_search_items()
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrGeo_db_search_params_listener($_data, $_user, $_conf, $_args, $event)
{
    // zip_code_radius = 20 (required, in miles)
    // zip_code_country = US (default)
    // profile_zip_code = 98036
    // profile_zip_code = 98036 || profile_zip_code = 98011
    if (!empty($_data['zip_code_radius']) && isset($_data['search']) && is_array($_data['search'])) {
        foreach ($_data['search'] as $k => $v) {
            if (strpos($v, '_zip_code')) {
                // We have a ZIP Code radius search
                $country = 'US';
                if (!empty($_data['zip_code_country']) && strlen($_data['zip_code_country']) === 2) {
                    $country = $_data['zip_code_country'];
                }
                if (strpos($v, ' || ')) {
                    // This is an OR condition
                    $_tmp = array();
                    foreach (explode(' || ', $v) as $cond) {
                        if (strpos($cond, 'zip_code')) {
                            @list($key, , $val) = explode(' ', $cond, 3);
                            $zip = trim($val);
                            if ($_zp = jrGeo_get_info_for_zip_code($zip, $country)) {
                                if ($_in = jrGeo_get_zip_codes_with_radius($_zp['zip_lat'], $_zp['zip_lon'], $_data['zip_code_radius'], $zip, $country)) {
                                    $_tmp[] = "{$key} in " . implode(',', array_keys($_in));
                                }
                            }
                        }
                        else {
                            $_tmp[] = $cond;
                        }
                    }
                    $_data[$k] = implode(' || ', $_tmp);
                    unset($_tmp);
                }
                else {
                    @list($key, , $val) = explode(' ', $v, 3);
                    $zip = trim($val);
                    if ($_zp = jrGeo_get_info_for_zip_code($zip, $country)) {
                        if ($_in = jrGeo_get_zip_codes_with_radius($_zp['zip_lat'], $_zp['zip_lon'], $_data['zip_code_radius'], $zip, $country)) {
                            $_data[$k] = "{$key} in " . implode(',', array_keys($_in));
                        }
                    }
                }
            }
        }
    }
    return $_data;
}

/**
 * Make sure City data file is uploaded
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrGeo_system_check_listener($_data, $_user, $_conf, $_args, $event)
{
    // Check for database files
    $url = jrCore_get_module_url('jrGeo');
    $_ch = jrGeo_get_available_database_info();
    foreach ($_ch as $c) {
        if (!jrCore_get_config_value('jrGeo', $c[0], false)) {
            $dat             = array();
            $dat[1]['title'] = $c[1];
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = 'uploaded';
            $dat[2]['class'] = 'center';
            $dat[3]['title'] = $_args['fail'];
            $dat[4]['title'] = "Geo Location database file is missing<br><a href=\"{$_conf['jrCore_base_url']}/{$url}/database\">Click here to install and enable</a>";
            $dat[3]['class'] = 'center';
            jrCore_page_table_row($dat);
        }
    }
    return $_data;
}

/**
 * update the geocode lat and lng when zip changes
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrGeo_db_update_item_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_args['module'])) {
        $pfx = jrCore_db_get_prefix($_args['module']);
        if (isset($_data["{$pfx}_zip"]) && strlen($_data["{$pfx}_zip"]) > 0) {
            if ($_zp = jrGeo_get_info_for_zip_code($_data["{$pfx}_zip"])) {
                $_data["{$pfx}_geocode_lat"] = $_zp['zip_lat'];
                $_data["{$pfx}_geocode_lng"] = $_zp['zip_lon'];
            }
        }

    }
    return $_data;
}

/**
 * SQL changes
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrGeo_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // Update Guam, Puerto Rico and US Virgin Islands to be "US"
    $tbl = jrCore_db_table_name('jrGeo', 'zip_code');
    $req = "UPDATE {$tbl} SET zip_country = 'US' WHERE zip_country IN('GU','PR','VI')";
    jrCore_db_query($req);
    return $_data;
}

/**
 * Signup Validate
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrGeo_signup_validate_listener($_data, $_user, $_conf, $_args, $event)
{
    // Watched during Signup
    global $_post;
    if (jrCore_module_is_active('jrBanned')) {
        if ($_config = jrBanned_get_banned_config('banned_country')) {
            if ($ip = jrCore_get_ip()) {
                if ($_lc = jrGeo_location($ip)) {
                    if (!empty($_lc['country_name'])) {
                        foreach ($_config as $v) {
                            if (stripos(' ' . $_lc['country_name'], $v)) {
                                if (jrCore_get_config_value('jrBanned', 'log_block', 'on') == 'on') {
                                    jrCore_logger('MAJ', "banned: blocked user signup country: {$_lc['country_name']}", array('_geo' => $_lc, '_post' => $_post));
                                }
                                jrCore_set_form_notice('error', 33);
                                jrCore_form_field_hilight('user_name');
                                jrCore_form_result();
                            }
                        }
                    }
                }
            }
        }
    }
    return $_data;
}


//-----------------------------
// Functions
//-----------------------------

/**
 * Return true if a viewing session is a banned bot
 * @param $value string
 * @param $_config array country names that have been banned
 * @return bool
 */
function jrGeo_is_banned_country($value, $_config)
{
    // Here as a stub to show up in Banned options
    return false;
}

/**
 * Get the ZIP code components passed in on the URL
 * @return array
 */
function jrGeo_get_url_components()
{
    global $_urls, $_post;
    // URL will be in one of TWO formats:
    // site.com/geo/zip_search/profile/profile_zip_code/US/98036/20  (with country code)
    // site.com/geo/zip_search/profile/profile_zip_code/98036/20     (without country code)
    if (!empty($_post['_3']) && strlen($_post['_3']) === 2) {
        // We have our COUNTRY CODE
        $_parts = array(
            'module'   => $_urls["{$_post['_1']}"],
            'field'    => $_post['_2'],
            'country'  => $_post['_3'],
            'zip_orig' => $_post['_4'],
            'zip_code' => $_post['_4'],
            'radius'   => intval($_post['_5'])
        );
    }
    else {
        $_parts = array(
            'module'   => $_urls["{$_post['_1']}"],
            'field'    => $_post['_2'],
            'country'  => jrGeo_get_country_code_from_zip_code($_post['_3']),
            'zip_orig' => $_post['_3'],
            'zip_code' => $_post['_3'],
            'radius'   => intval($_post['_4'])
        );
    }
    if ($_parts['country'] == 'CA') {
        // For canada we only use the first THREE digits of the zip code
        $_parts['zip_code'] = substr($_parts['zip_code'], 0, 3);
    }
    return $_parts;
}

/**
 * get country code from passed in ZIP code
 * @param string $zip ZIP Code
 * @return string
 */
function jrGeo_get_country_code_from_zip_code($zip)
{
    $country = 'US';
    if (is_numeric($zip)) {
        if (strlen($zip) === 4) {
            $country = 'NZ';
        }
    }
    else {
        $zip = str_replace(' ', '', $zip);
        if (strlen($zip) <= 6) {
            if (strlen($zip) === 6 || strlen($zip) === 3) {
                // could be canadian
                $country = 'CA';
            }
        }
    }
    return $country;
}

/**
 * Get information about available GEO databases
 * @return array
 */
function jrGeo_get_available_database_info()
{
    return array(
        'geoip2city' => array(
            'ip2_file_time',
            'Geo Location Database - version 2',
            'https://www.jamroom.net/r/geoip2-database-download',
            'mmdb',
            'GeoLite 2 Location database provided by <a href="http://www.maxmind.com"><u>MaxMind</u></a>'
        ),
        'geoipcity'  => array(
            'ip_file_time',
            'Geo Location Database - version 1 (deprecated)',
            'https://www.jamroom.net/r/geoip-database-download',
            'dat',
            'GeoLite Location database provided by <a href="http://www.maxmind.com"><u>MaxMind</u></a>'
        ),
        'zipcode'    => array(
            'zip_file_time',
            'ZIP Code Database',
            'https://www.jamroom.net/r/zip-code-database-download',
            'txt',
            'ZIP Code database provided by <a href="http://www.geonames.org"><u>GeoNames</u></a>'
        )
    );
}

/**
 * Load the ZIP file database
 * @param string $file
 * @return bool
 */
function jrGeo_load_zip_database($file)
{
    @ini_set('memory_limit', '256M');
    if ($h = fopen($file, 'r')) {

        // Clean out old entries
        $tbl = jrCore_db_table_name('jrGeo', 'zip_code');
        $req = "TRUNCATE TABLE {$tbl}";
        jrCore_db_query($req);

        $tot = 0;
        $cnt = 0;
        $_in = array();
        while (!feof($h)) {
            if ($line = fgets($h)) {
                if ($_zp = explode("\t", $line)) {
                    $place = jrCore_db_escape("{$_zp[2]}, {$_zp[4]}");
                    $_in[] = "('{$_zp[0]}','{$_zp[1]}','{$place}','{$_zp[9]}','{$_zp[10]}')";
                    $cnt++;
                    if (($cnt % 1000) === 0 && $cnt > 0) {
                        $req = "INSERT INTO {$tbl} VALUES " . implode(',', $_in);
                        jrCore_db_query($req);
                        $tot += count($_in);
                        $_in = array();
                        $cnt = 0;
                    }
                }
            }
        }
        fclose($h);

        // Pick up any stragglers
        if (count($_in) > 0) {
            $req = "INSERT INTO {$tbl} VALUES " . implode(',', $_in);
            jrCore_db_query($req);
            $tot += count($_in);
        }
        if ($tot > 0) {
            jrCore_logger('INF', 'successully updated ' . jrCore_number_format($tot) . ' ZIP Code database entries');
        }
        return true;
    }
    @fclose($h);

    return false;
}

/**
 * Get GeoIP lookup info for an IP address
 * @param string $ip IP Address
 * @param bool $force_local Set to TRUE to force local lookup of IP from ip data file even if MaxMind API is enabled
 * @return array|bool
 */
function jrGeo_location($ip, $force_local = false)
{
    if (!$ip || strlen($ip) < 7 || strlen($ip) > 15) {
        // Quick check on length fails
        return false;
    }

    // Have we already seen this one in this process?
    $key = 'jrgeo_checked_ips';
    if ($_cc = jrCore_get_flag($key)) {
        if (isset($_cc[$ip])) {
            return $_cc[$ip];
        }
    }
    else {
        $_cc = array();
    }

    if (!jrCore_checktype($ip, 'ip_address')) {
        // Not an IP
        return false;
    }
    if (jrCore_checktype($ip, 'private_ip_address')) {
        // Private IP
        return false;
    }

    // REQUIRED plugin return FORMAT:
    // [country_code] => US
    // [country_name] => United States
    // [region] => WA
    // [city] => Ferndale
    // [postal_code] => 98248
    // [latitude] => 48.8645
    // [longitude] => -122.6307
    // [dma_code] => 819
    // [metro_code] => 819
    // [continent_code] => NA

    $act = jrCore_get_config_value('jrGeo', 'active', 'local');
    if ($act == 'local') {
        // We are set for "local" - we may be an upgrade but have no DB yet - use "old" for now
        if (!jrCore_get_config_value('jrGeo', 'ip2_file_time', false)) {
            $act = 'old';
        }
    }
    $fnc = "jrGeo_get_ip_location_data_{$act}";
    if (!function_exists($fnc)) {
        return false;
    }
    if ($_rt = $fnc($ip)) {
        // add to flag cache
        $_cc[$ip] = $_rt;
        jrCore_set_flag($key, $_cc);
        return $_rt;
    }
    return false;
}

/**
 * "api_lite" - GeoLite2 API
 * @param string $ip IP Address to get data about
 * @return array|false
 */
function jrGeo_get_ip_location_data_apilite($ip)
{
    if (jrGeo_api_is_configured()) {
        if (!$_rt = jrGeo_get_cached_ip_info($ip)) {
            require_once APP_DIR . '/modules/jrGeo/contrib/geoip2/autoload.php';
            $account = jrCore_get_config_value('jrGeo', 'user_id', false);
            $api_key = jrCore_get_config_value('jrGeo', 'license_key', false);
            try {
                $client = new Client($account, $api_key, array('en'), array('host' => 'geolite.info'));
                $record = $client->city($ip);
            }
            catch (Exception $e) {
                return false;
            }
            if ($_rs = $record->raw) {
                return array(
                    'country_code'   => (isset($_rs['country']['iso_code'])) ? $_rs['country']['iso_code'] : '',
                    'country_name'   => (isset($_rs['country']['names']['en'])) ? $_rs['country']['names']['en'] : '',
                    'region'         => (isset($_rs['subdivisions'][0]['iso_code'])) ? $_rs['subdivisions'][0]['iso_code'] : '',
                    'city'           => (isset($_rs['city']['names']['en'])) ? $_rs['city']['names']['en'] : '',
                    'postal_code'    => (isset($_rs['postal']['code'])) ? $_rs['postal']['code'] : '',
                    'latitude'       => (isset($_rs['location']['latitude'])) ? $_rs['location']['latitude'] : '',
                    'longitude'      => (isset($_rs['location']['longitude'])) ? $_rs['location']['longitude'] : '',
                    'dma_code'       => (isset($_rs['location']['metro_code'])) ? $_rs['location']['metro_code'] : '',
                    'metro_code'     => (isset($_rs['location']['metro_code'])) ? $_rs['location']['metro_code'] : '',
                    'continent_code' => (isset($_rs['continent']['code'])) ? $_rs['continent']['code'] : ''
                );
            }
        }
    }
    return false;
}

/**
 * "api" - Geo Precision API
 * @param string $ip IP Address to get data about
 * @return array|false
 */
function jrGeo_get_ip_location_data_api($ip)
{
    if (jrGeo_api_is_configured()) {
        if (!$_rt = jrGeo_get_cached_ip_info($ip)) {
            $url = 'https://geoip.maxmind.com/geoip/v2.1/city/' . $ip;
            $uid = jrCore_get_config_value('jrGeo', 'user_id', '');
            $key = jrCore_get_config_value('jrGeo', 'license_key', '');
            $_rs = jrCore_load_url($url, null, 'GET', 443, $uid, $key, false, 6);
            if (strlen(trim($_rs)) > 0) {
                if ($_rs = json_decode($_rs, true)) {
                    $_rt = array(
                        'country_code'   => (isset($_rs['country']['iso_code'])) ? $_rs['country']['iso_code'] : '',
                        'country_name'   => (isset($_rs['country']['names']['en'])) ? $_rs['country']['names']['en'] : '',
                        'region'         => (isset($_rs['subdivisions'][0]['iso_code'])) ? $_rs['subdivisions'][0]['iso_code'] : '',
                        'city'           => (isset($_rs['city']['names']['en'])) ? $_rs['city']['names']['en'] : '',
                        'postal_code'    => (isset($_rs['postal']['code'])) ? $_rs['postal']['code'] : '',
                        'latitude'       => (isset($_rs['location']['latitude'])) ? $_rs['location']['latitude'] : '',
                        'longitude'      => (isset($_rs['location']['longitude'])) ? $_rs['location']['longitude'] : '',
                        'dma_code'       => (isset($_rs['location']['metro_code'])) ? $_rs['location']['metro_code'] : '',
                        'metro_code'     => (isset($_rs['location']['metro_code'])) ? $_rs['location']['metro_code'] : '',
                        'continent_code' => (isset($_rs['continent']['code'])) ? $_rs['continent']['code'] : ''
                    );
                    jrGeo_save_cached_ip_info($ip, $_rt);
                }
            }
        }
        return (is_array($_rt)) ? $_rt : false;
    }
    return false;
}

/**
 * "local" (v2) GeoIP lookup function
 * @param string $ip IP Address to get data about
 * @return array|false
 */
function jrGeo_get_ip_location_data_local($ip)
{
    if (!$time = jrCore_get_config_value('jrGeo', 'ip2_file_time', false)) {
        // Not setup yet
        return false;
    }
    $dir = jrCore_get_media_directory(0, FORCE_LOCAL);
    $fil = "{$dir}/geoip2city-{$time}.mmdb";
    if (!is_file($fil)) {
        // We don't have this one - could be a new DAT file
        // Cleanup any "old" DAT files that could be hanging around
        $_fl = glob("{$dir}/geoip2city-*");
        if (is_array($_fl)) {
            foreach ($_fl as $file) {
                jrCore_unlink($file);
            }
        }
        if (!jrCore_confirm_media_file_is_local(0, 'geoip2city.mmdb', $fil)) {
            // We don't exist
            return false;
        }
    }
    require_once APP_DIR . '/modules/jrGeo/contrib/geoip2/autoload.php';
    try {
        $reader = new Reader($fil);
        $record = $reader->city($ip);
    }
    catch (Exception $e) {
        return false;
    }
    if ($_rs = $record->raw) {
        return array(
            'country_code'   => (isset($_rs['country']['iso_code'])) ? $_rs['country']['iso_code'] : '',
            'country_name'   => (isset($_rs['country']['names']['en'])) ? $_rs['country']['names']['en'] : '',
            'region'         => (isset($_rs['subdivisions'][0]['iso_code'])) ? $_rs['subdivisions'][0]['iso_code'] : '',
            'city'           => (isset($_rs['city']['names']['en'])) ? $_rs['city']['names']['en'] : '',
            'postal_code'    => (isset($_rs['postal']['code'])) ? $_rs['postal']['code'] : '',
            'latitude'       => (isset($_rs['location']['latitude'])) ? $_rs['location']['latitude'] : '',
            'longitude'      => (isset($_rs['location']['longitude'])) ? $_rs['location']['longitude'] : '',
            'dma_code'       => (isset($_rs['location']['metro_code'])) ? $_rs['location']['metro_code'] : '',
            'metro_code'     => (isset($_rs['location']['metro_code'])) ? $_rs['location']['metro_code'] : '',
            'continent_code' => (isset($_rs['continent']['code'])) ? $_rs['continent']['code'] : ''
        );
    }
    return false;
}

/**
 * "old" (v1) GeoIP lookup function
 * @param string $ip IP Address to get data about
 * @return array|false
 */
function jrGeo_get_ip_location_data_old($ip)
{
    if (!$time = jrCore_get_config_value('jrGeo', 'ip_file_time', false)) {
        // Not setup yet
        return false;
    }
    $dir = jrCore_get_media_directory(0, FORCE_LOCAL);
    $fil = "{$dir}/geoipcity-{$time}.dat";
    if (!is_file($fil)) {
        // We don't have this one - could be a new DAT file
        // Cleanup any "old" DAT files that could be hanging around
        $_fl = glob("{$dir}/geoipcity-*");
        if (is_array($_fl)) {
            foreach ($_fl as $file) {
                jrCore_unlink($file);
            }
        }
        if (!jrCore_confirm_media_file_is_local(0, 'geoipcity.dat', $fil)) {
            // We don't exist
            return false;
        }
    }
    require_once APP_DIR . '/modules/jrGeo/contrib/geoip/geoipcity.php';
    $geo = geoip_open($fil, GEOIP_STANDARD);
    $_rt = (array) GeoIP_record_by_addr($geo, $ip);
    geoip_close($geo);
    if ($_rt) {
        unset($_rt['country_code3'], $_rt['area_code']);
        return $_rt;
    }
    return false;
}

/**
 * Get cached IP info for an IP
 * @param string $ip
 * @return bool|mixed
 */
function jrGeo_get_cached_ip_info($ip)
{
    $tbl = jrCore_db_table_name('jrGeo', 'ip_cache');
    $req = "SELECT ip_time, ip_info FROM {$tbl} WHERE ip_address = '" . jrCore_db_escape($ip) . "' LIMIT 1";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if ($_rt && is_array($_rt)) {
        return json_decode($_rt['ip_info'], true);
    }
    return false;
}

/**
 * Save IP info to the IP cache
 * @param string $ip
 * @param array $_info
 * @return bool
 */
function jrGeo_save_cached_ip_info($ip, $_info)
{
    $tbl = jrCore_db_table_name('jrGeo', 'ip_cache');
    $ipa = jrCore_db_escape($ip);
    $inf = jrCore_db_escape(json_encode($_info));
    $req = "INSERT INTO {$tbl} (ip_address, ip_time, ip_info) VALUES ('{$ipa}', UNIX_TIMESTAMP(), '{$inf}')
            ON DUPLICATE KEY UPDATE ip_time = UNIX_TIMESTAMP(), ip_info = VALUES(ip_info)";
    $cnt = jrCore_db_query($req, 'COUNT');
    if ($cnt > 0) {
        return true;
    }
    return false;
}

/**
 * Return TRUE if API is configured
 * @return bool
 */
function jrGeo_api_is_configured()
{
    if (jrCore_get_config_value('jrGeo', 'license_key', false)) {
        if (jrCore_get_config_value('jrGeo', 'user_id', false)) {
            return true;
        }
    }
    return false;
}

/**
 * Get distance between two coordinates
 * @param $latitude1 string
 * @param $longitude1 string
 * @param $latitude2 string
 * @param $longitude2 string
 * @return array
 */
function jrGeo_distance_between_points($latitude1, $longitude1, $latitude2, $longitude2)
{
    $theta = ($longitude1 - $longitude2);
    $miles = (sin(deg2rad($latitude1)) * sin(deg2rad($latitude2))) + (cos(deg2rad($latitude1)) * cos(deg2rad($latitude2)) * cos(deg2rad($theta)));
    $miles = acos($miles);
    $miles = rad2deg($miles);
    $miles = ($miles * 60 * 1.1515);
    return array(
        'miles'      => round($miles, 2),
        'feet'       => round($miles * 5280, 2),
        'kilometers' => round($miles * 1.609344, 2),
        'meters'     => round(($miles * 1.609344) * 1000, 2)
    );
}

/**
 * Get the stored info (lat, lon, etc) for a ZIP code
 * @param string $zip
 * @param string $country
 * @return array|false
 */
function jrGeo_get_info_for_zip_code($zip, $country = 'US')
{
    if (!empty($zip)) {
        $tbl = jrCore_db_table_name('jrGeo', 'zip_code');
        $req = "SELECT * FROM {$tbl} WHERE zip_code = '" . jrCore_db_escape($zip) . "' AND zip_country = '" . jrCore_db_escape($country) . "'";
        return jrCore_db_query($req, 'SINGLE');
    }
    return false;
}

/**
 * Get locations within a ZIP Code radius (miles)
 * @param string $latitude Latitude
 * @param string $longitude Longitude
 * @param int $radius Radius distance
 * @param string $zip_code Used for Caching
 * @param string $country optional - speeds up search
 * @return array|false
 */
function jrGeo_get_zip_codes_with_radius($latitude, $longitude, $radius, $zip_code = '', $country = null)
{
    if (empty($latitude) || empty($longitude) || empty($radius)) {
        return false;
    }
    $key = json_encode(func_get_args());
    if (!$_rt = jrCore_is_cached('jrGeo', $key, false, false)) {
        $lat = $latitude;
        $lon = $longitude;
        $rad = (int) $radius;
        $fmt = (jrCore_get_config_value('jrGeo', 'zip_format', 'miles') == 'miles') ? 3958 : 2459;
        $tbl = jrCore_db_table_name('jrGeo', 'zip_code');
        $req = "SELECT *, ROUND(({$fmt} * 3.1415926 * sqrt((zip_lat - {$lat}) * (zip_lat - {$lat}) + cos(zip_lat / 57.29578) * cos({$lat} / 57.29578) * (zip_lon - {$lon}) * (zip_lon- {$lon})) / 180), 1) AS distance
                  FROM {$tbl} WHERE ({$fmt} * 3.1415926 * sqrt((zip_lat - {$lat}) * (zip_lat - {$lat}) + cos(zip_lat / 57.29578) * cos({$lat} / 57.29578) * (zip_lon - {$lon}) * (zip_lon- {$lon})) / 180) <= {$rad}";
        if (!empty($country) && strlen($country) === 2) {
            $req .= " AND zip_country = '" . strtoupper($country) . "'";
        }
        $_rt = jrCore_db_query($req, 'zip_code');
        if (!$_rt || !is_array($_rt)) {
            $_rt = 'no_results';
        }
        else {
            uasort($_rt, function ($a, $b) {
                return ($a['distance'] > $b['distance']) ? 1 : -1;
            });
        }
        jrCore_add_to_cache('jrGeo', $key, $_rt, 0, 0, false, false);
        if (!empty($country) && !empty($zip_code)) {
            jrGeo_add_cached_radius_info($country, $zip_code, $radius, $_rt);
        }
    }
    return (is_array($_rt)) ? $_rt : false;
}

/**
 * Add cached radius info
 * @param string $country
 * @param string $zip_code
 * @param int $radius
 * @param array $_data
 * @return false|int
 */
function jrGeo_add_cached_radius_info($country, $zip_code, $radius, $_data)
{
    $fmt = jrCore_get_config_value('jrGeo', 'zip_format', 'miles');
    $key = md5("{$country}/{$zip_code}/{$radius}/{$fmt}");
    $dat = jrCore_db_escape(json_encode($_data));
    $tbl = jrCore_db_table_name('jrGeo', 'zip_cache');
    $req = "INSERT IGNORE INTO {$tbl} (zip_hash, zip_info) VALUES ('{$key}', COMPRESS('{$dat}'))";
    return jrCore_db_query($req, 'COUNT');
}

/**
 * Get cached radius info for a zip code
 * @param string $country
 * @param string $zip_code
 * @param int $radius
 * @return array|false
 */
function jrGeo_get_cached_radius_info($country, $zip_code, $radius)
{
    $fmt = jrCore_get_config_value('jrGeo', 'zip_format', 'miles');
    $key = md5("{$country}/{$zip_code}/{$radius}/{$fmt}");
    $tbl = jrCore_db_table_name('jrGeo', 'zip_cache');
    $req = "SELECT UNCOMPRESS(zip_info) AS zip_info FROM {$tbl} WHERE zip_hash = '{$key}'";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if ($_rt && is_array($_rt) && !empty($_rt['zip_info'])) {
        return json_decode($_rt['zip_info'], true);
    }
    return false;
}

/**
 * Get a physical address from an item's data
 * @param string $module
 * @param array $_item
 * @return string
 */
function jrGeo_get_address_from_item_data($module, $_item)
{
    $pfx = jrCore_db_get_prefix($module);
    if (isset($_item["{$pfx}_address"])) {
        return $_item["{$pfx}_address"];
    }
    $_ad = array();
    $_ch = array(
        'address1',
        'address2',
        'address_line1',
        'address_line2',
        'city',
        'state',
        'zip_code',
        'country'
    );
    foreach ($_ch as $v) {
        if (isset($_item["{$pfx}_{$v}"])) {
            $_ad[] = $_item["{$pfx}_{$v}"];
        }
    }
    if (count($_ad) > 0) {
        return implode(' ', $_ad);
    }
    return false;
}

/**
 * Get the latitude and longitude for a physical address
 * @param string $address
 * @return array|false
 */
function jrGeo_get_geocode_for_address($address)
{
    // https://maps.googleapis.com/maps/api/geocode/outputFormat?parameters
    if ($key = jrGeo_get_google_api_key()) {
        $url = "https://maps.googleapis.com/maps/api/geocode/json?key={$key}&address=" . urlencode(trim($address));
        $res = jrCore_load_url($url);
        if ($res) {
            if ($res = json_decode($res, true)) {
                if (!empty($res['results'][0]['geometry'])) {
                    return array(
                        'latitude'  => $res['results'][0]['geometry']['location']['lat'],
                        'longitude' => $res['results'][0]['geometry']['location']['lng'],
                        'place_id'  => $res['results'][0]['place_id']
                    );
                }
            }
        }
    }
    return false;
}

/**
 * Get the Google API Key for Maps
 * @return bool
 */
function jrGeo_get_google_api_key()
{
    return jrCore_get_config_value('jrGeo', 'google_api_key', false);
}

/**
 * Get the ZIP datastore keys for a module
 * @param string $mod
 * @return array|bool|mixed
 */
function jrGeo_get_zip_keys_for_module($mod)
{
    $key = "{$mod}_zip_keys";
    if (!$_zp = jrCore_is_cached('jrGeo', $key, false, false)) {
        // Get unique DataStore keys
        if ($_mi = jrCore_db_get_unique_keys($mod)) {
            foreach ($_mi as $v) {
                if (strpos($v, '_zip_code')) {
                    if (!$_zp) {
                        $_zp = array(array());
                    }
                    $_zp[0][] = $v;
                }
            }
        }
        if (!$_zp) {
            return false;
        }
        jrCore_add_to_cache('jrGeo', $key, $_zp, 0, 0, false, false);
    }
    return $_zp;
}

//-----------------------------
// Smarty
//-----------------------------

/**
 * Distance
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrGeo_distance($params, $smarty)
{
    if (!isset($params['ip1'])) {
        return jrCore_smarty_missing_error('ip1');
    }
    if (!isset($params['ip2'])) {
        return jrCore_smarty_missing_error('ip2');
    }
    if ($_ip1 = jrGeo_location($params['ip1'])) {
        if ($_ip2 = jrGeo_location($params['ip2'])) {
            $_rt = jrGeo_distance_between_points($_ip1['latitude'], $_ip1['longitude'], $_ip2['latitude'], $_ip2['longitude']);
            if (isset($_rt) && is_array($_rt)) {
                // Check for template
                if (!empty($params['template'])) {
                    if (strpos($params['template'], '.tpl')) {
                        $out = jrCore_parse_template($params['template'], $_rt);
                    }
                    else {
                        $_rp = array();
                        foreach ($_rt as $k => $v) {
                            $_rp["%{$k}%"] = $v;
                        }
                        $out = str_replace(array_keys($_rp), $_rp, $params['template']);
                        unset($_rp, $_rt);
                    }
                }
                else {
                    $out = $_rt['miles'];
                }
                if (!empty($params['assign'])) {
                    $smarty->assign($params['assign'], $out);
                    return '';
                }
                return $out;
            }
        }
    }
    return '';
}

/**
 * Location
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrGeo_location($params, $smarty)
{
    if (!isset($params['ip'])) {
        $params['ip'] = jrCore_get_ip();
    }
    $_rt = jrGeo_location($params['ip']);
    if (!$_rt) {
        return '';
    }

    // [country_code] => US
    // [country_code3] => USA
    // [country_name] => United States
    // [region] => WA
    // [city] => Bothell
    // [postal_code] => 98021
    // [latitude] => 47.7948
    // [longitude] => -122.2054
    // [area_code] => 425
    // [dma_code] => 819
    // [metro_code] => 819
    // [continent_code] => NA

    $out = '';
    // Check for template
    if (!empty($params['template'])) {
        if (strpos($params['template'], '.tpl')) {
            $out = jrCore_parse_template($params['template'], $_rt);
        }
        else {
            $_rp = array();
            foreach ($_rt as $k => $v) {
                $_rp["%{$k}%"] = $v;
            }
            $out = str_replace(array_keys($_rp), $_rp, $params['template']);
            unset($_rp, $_rt);
        }
    }
    else {
        $_tm   = array();
        $_tm[] = (isset($_rt['city'])) ? $_rt['city'] : '';
        $_tm[] = (isset($_rt['region'])) ? $_rt['region'] : '';
        $_tm[] = (isset($_rt['country_name'])) ? $_rt['country_name'] : '';
        if (count($_tm) > 0) {
            $out = implode(', ', $_tm);
        }
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Search Form
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrGeo_search_form($params, $smarty)
{
    if (!isset($params['module'])) {
        jrCore_smarty_missing_error('module');
    }
    if (!isset($params['field'])) {
        jrCore_smarty_missing_error('field');
    }
    if (!isset($params['ip'])) {
        $params['ip'] = jrCore_get_ip();
    }
    $_rep = array(
        'function' => 'jrGeo_search_form',
        'params'   => $params,
        '_rt'      => jrGeo_location($params['ip']),
        'radius'   => (!empty($params['radius']) && jrCore_checktype($params['radius'], 'number_nz')) ? intval($params['radius']) : 25
    );

    $out = jrCore_parse_template('zip_search_form.tpl', $_rep, 'jrGeo');

    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}
