<?php
 /**
 * Jamroom Newsletters module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2021 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * view: compose_save
 * @param array $_post Posted Data
 * @param array $_user Viewing User data
 * @param array $_conf Global Config
 */
function view_jrNewsLetter_compose_save($_post, $_user, $_conf)
{
    // Must be logged in as admin
    jrUser_session_require_login();
    jrUser_master_only();

    // Save off letter_message before form_validate - prevents any issues with invalid HTML
    $msg = (isset($_post['letter_message'])) ? $_post['letter_message'] : '';
    $sch = (isset($_post['letter_scheduled'])) ? $_post['letter_scheduled'] : '';

    jrCore_set_flag('master_html_trusted', true); // Trust HTML input from master user for newsletter
    jrCore_form_validate($_post);

    // quotas and filters will come in as a comma separated string
    $_quotas = explode(',', $_post['letter_quota']);
    $_filter = explode(',', $_post['letter_filter']);
    $_custom = array();
    foreach ($_post as $k => $v) {
        if (strpos($k, 'letter_custom') === 0 && !empty($v)) {
            $tmp = trim(preg_replace('/\s\s+/', ' ', $v));
            list(, $operator,) = explode(' ', $tmp);
            switch (strtolower($operator)) {
                case '=':
                case '>':
                case '<':
                case '>=':
                case '&lt;=':
                case 'in':
                case 'not_in':
                case 'like':
                case 'not_like':
                    break;
                default:
                    jrCore_form_field_hilight($k);
                    jrCore_set_form_notice('error', 'Invalid Custom Filter Field');
                    jrCore_form_result();
                    break;
            }
            $_custom[] = trim($v);
        }
    }

    // Save newsletter to DS
    $_tmp = array(
        'letter_sent'      => 'UNIX_TIMESTAMP()',
        'letter_title'     => $_post['letter_title'],
        'letter_message'   => $msg,
        'letter_quota'     => json_encode($_quotas),
        'letter_filter'    => json_encode($_filter),
        'letter_custom'    => json_encode($_custom),
        'letter_scheduled' => $sch
    );

    $_post['letter_custom'] = $_tmp['letter_custom'];

    // Is this the admin sending a test to themselves?

    foreach ($_quotas as $quota) {
        if (strpos(' ' . $quota, 'test_')) {
            $_tmp['letter_draft'] = 1;
            // This is a TEST mailing of existing newsletter - do not create NEW - update existing
            if (isset($_post['letter_id']) && jrCore_checktype($_post['letter_id'], 'number_nz')) {
                $nid = (int) $_post['letter_id'];
                if (!jrCore_db_update_item('jrNewsLetter', $nid, $_tmp)) {
                    jrCore_set_form_notice('error', 'an error was encountered updating the newsletter in the DataStore');
                    jrCore_location('referrer');
                }
            }
            // This is a TEST send - create if the first time
            else {
                $_core = array(
                    '_profile_id' => jrUser_get_profile_home_key('_profile_id')
                );
                if (!$nid = jrCore_db_create_item('jrNewsLetter', $_tmp, $_core)) {
                    jrCore_set_form_notice('error', 'an error was encountered creating the newsletter in the DataStore');
                    jrCore_location('referrer');
                }
            }
            switch ($quota) {
                case 'test_user':
                    // Send newsletter to admin
                    // If we are currently UN SUBSCRIBED - make sure we're re-subscribed
                    if (isset($_user['user_jrNewsLetter_newsletter_notifications']) && $_user['user_jrNewsLetter_newsletter_notifications'] == 'off') {
                        jrCore_db_update_item('jrUser', $_user['_user_id'], array('user_jrNewsLetter_newsletter_notifications' => 'email'));
                    }
                    list($subject, $message) = jrNewsLetter_replace_user_variables($_post['letter_title'], $msg, $_user);
                    jrUser_notify($_user['_user_id'], 0, 'jrNewsLetter', 'newsletter', $subject, $message);
                    break;
                case 'test_master':
                case 'test_admin':
                    if ($_post['letter_quota'] == 'test_master') {
                        // Send newsletter to all master users
                        $_us = jrUser_get_master_user_ids();
                    }
                    else {
                        // Send newsletter to all master + admin users
                        $_us = jrUser_get_admin_user_ids();
                    }
                    if ($_us && is_array($_us)) {
                        foreach ($_us as $uid) {
                            $_usr = jrCore_db_get_item('jrUser', $uid);
                            if ($_usr && is_array($_usr)) {
                                // If we are currently UN SUBSCRIBED - make sure we're re-subscribed
                                if (isset($_usr['user_jrNewsLetter_newsletter_notifications']) && $_usr['user_jrNewsLetter_newsletter_notifications'] == 'off') {
                                    jrCore_db_update_item('jrUser', $uid, array('user_jrNewsLetter_newsletter_notifications' => 'email'));
                                }
                                list($subject, $message) = jrNewsLetter_replace_user_variables($_post['letter_title'], $msg, $_usr);
                                jrUser_notify($uid, 0, 'jrNewsLetter', 'newsletter', $subject, $message);
                            }
                        }
                    }
                    break;
            }
            jrCore_form_delete_session();
            jrCore_set_form_notice('success', "A TEST NewsLetter was successfully sent");
            jrCore_location("{$_conf['jrCore_base_url']}/{$_post['module_url']}/compose/draft={$nid}");
        }
    }

    // Check that this newsletter is not scheduled
    if (isset($_post['letter_scheduled']) && jrCore_checktype($_post['letter_scheduled'], 'number_nz')) {
        if ($_post['letter_scheduled'] >= time()) {
            jrCore_set_form_notice('error', 'This Newsletter has been scheduled to be sent on a future date. Save it by using the \'Save Changes/Draft\' button');
            jrCore_location('referrer');
        }
    }
    unset ($_post['letter_scheduled']);

    // FALL THROUGH - sending for real!

    // Auto save as template if configured
    if (isset($_conf['jrNewsLetter_auto_tpl']) && $_conf['jrNewsLetter_auto_tpl'] == 'on') {
        $template_title = $_post['letter_title'] . ' (' . date('d M Y h:i:sA') . ')';
        jrNewsLetter_save_template($template_title, $msg);
    }

    // Create new newsletter ID so we start a new campaign
    if (isset($_post['letter_id']) && jrCore_checktype($_post['letter_id'], 'number_nz')) {
        $nid = $_post['letter_id'];
    }
    else {
        if (!$nid = jrCore_db_create_item('jrNewsLetter', $_tmp)) {
            jrCore_set_form_notice('error', 'an error was encountered creating the newsletter in the DataStore');
            jrCore_form_result();
        }
    }

    // Our newsletter
    $_post['newsletter_id'] = $nid;

    // Setup campaign
    $campaign             = jrMailer_get_campaign_id('jrNewsLetter', $nid, $_post['letter_title'], $msg);
    $_post['campaign_id'] = $campaign;

    // Update newsletter with campaign ID
    jrCore_db_update_item('jrNewsLetter', $nid, array('letter_campaign_id' => $campaign));

    // It's no longer a draft - remove draft key
    jrCore_db_delete_item_key('jrNewsLetter', $nid, 'letter_draft');

    // Submit to prep queue
    $_post['letter_message']     = $msg;
    $_post['letter_scheduled']   = $sch;
    $_post['notification_email'] = $_user['user_email'];
    jrCore_queue_create('jrNewsLetter', 'prep_newsletter', $_post);

    // redirect to campaign
    jrCore_set_form_notice('success', 'The NewsLetter has been submitted for delivery!<br>Refresh this page for the latest results - there is no need to keep your browser open.', false);
    jrCore_form_delete_session();
    $url = jrCore_get_module_url('jrMailer');
    jrCore_location("{$_conf['jrCore_base_url']}/{$url}/campaign_result/{$campaign}");
}
