<?php
 /**
 * Jamroom Payment Support module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2021 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * view: webhook
 * @param array $_post Posted Data
 * @param array $_user Viewing User data
 * @param array $_conf Global Config
 */
function view_jrPayment_webhook($_post, $_user, $_conf)
{
    // http://site.com/payment/webhook/[plugin]
    if (isset($_post['_1']) && jrPayment_is_valid_plugin($_post['_1'])) {
        // This is optional, but if we get the plugin on the webhook we use it -
        // that way if a system changes active payment plugins, previous webhooks
        // for things like refunds will continue to function properly
        jrPayment_set_active_plugin($_post['_1']);
    }

    // Step 1 - Parse and Validate
    // This ensures we have a valid callback to our web hook and the plugin
    // formats it to ensure we have some required pieces of info
    $_tx = jrPayment_run_plugin_function('webhook_parse', $_post);
    if ($_tx && is_array($_tx)) {

        // Give listening modules a chance to process this
        $_tx = jrCore_trigger_event('jrPayment', 'webhook_parsed', $_tx);
        if (isset($_tx['transaction_complete']) && $_tx['transaction_complete'] == 1) {
            // We were handled 100% by a listener
            jrPayment_run_plugin_function('webhook_response', 'success', $_post);
            header('HTTP/1.0 200 OK');
            exit;
        }

        // If we are using the FOXYCART plugin, then we need to watch for REFUND
        // transactions from other gateways, since FOXYCART does not handle refunds
        if ($_conf['jrPayment_plugin'] == 'foxycart' && $_post['_1'] != 'foxycart') {

            // This is a webhook from paypal/stripe, but FoxyCart is active.
            if (isset($_tx['txn_type']) && $_tx['txn_type'] != 'refund') {
                // This is NOT a refund transaction - exit as we handle this direct
                jrPayment_run_plugin_function('webhook_response', 'success', $_post);
                header('HTTP/1.0 200 OK');
                exit;
            }
        }

        // Did we get a valid transaction type?
        if (!isset($_tx['txn_type'])) {
            jrCore_logger('CRI', "Payments: txn_type not found in incoming transaction", array('_tx' => $_tx, '_post' => $_post));
            jrPayment_run_plugin_function('webhook_response', 'failure', $_post);
            header('HTTP/1.0 400 Bad Response');
            exit;
        }

        // Have we already processed this transaction?
        if (isset($_tx['txn_id'])) {
            if (jrCore_db_get_item_by_key('jrPayment', 'txn_id', $_tx['txn_id'], true)) {
                // We've already processed this transaction
                jrCore_logger('MAJ', "Payments: duplicate transaction received: {$_tx['txn_id']}", $_tx);
                jrPayment_run_plugin_function('webhook_response', 'success', $_post);
                header('HTTP/1.0 200 OK');
                exit;
            }
        }

        // Store our transaction in the DS
        $tid = jrPayment_create_transaction($_tx);

        // Save stats
        jrCore_create_stat_entry('jrPayment', 'txn_count', $_tx['txn_type'], 0, 0, null, false);
        if (isset($_tx['txn_total']) && $_tx['txn_total'] > 0) {
            jrCore_create_stat_entry('jrPayment', 'txn_total', $_tx['txn_type'], 0, 0, null, false, $_tx['txn_total']);
        }

        // Log incoming raw data
        $plug = jrPayment_get_active_plugin();
        if ($_tx['txn_type'] != 'information' && isset($_tx['txn_id']) && strlen($_tx['txn_id']) > 0) {
            jrCore_logger('DBG', "Payments: processing incoming {$plug} transaction: {$_tx['txn_id']}", $_tx);
        }
        $_tx['txn_plugin']   = $plug;
        $_tx['txn_currency'] = jrPayment_run_plugin_function('get_currency_code');

        // Did we get all required fields for a balance affecting transaction?
        if ($_tx['txn_type'] == 'payment' && !jrPayment_transaction_contains_required_fields($_tx)) {
            jrPayment_run_plugin_function('webhook_response', 'failure', $_post);
            header('HTTP/1.0 400 Bad Response');
            exit;
        }

        // process
        if ($tid && jrCore_checktype($tid, 'number_nz')) {

            // Step 2 - Process
            // Let the plugin process and remove unnecessary events
            $_tx['txn_item_id'] = $tid;
            $_tx                = jrPayment_run_plugin_function('webhook_process', $_tx);
            if ($_tx && is_array($_tx)) {

                // Trigger - allow modules to process their own items if needed
                $_tx = jrCore_trigger_event('jrPayment', 'webhook_event', $_tx);

                // Note: As long as the transaction does NOT have txn_cart_id
                // in it, then it can be for any other module/situation as
                // long as it is handled by the listener (i.e. subscriptions)
                // or in the payment webhook_process function

                //------------------------
                // POINT-OF-SALE PAYMENT
                //------------------------
                if (isset($_tx['txn_pos_payment']) && $_tx['txn_pos_payment'] == 1) {

                    $tag = null;
                    if (isset($_tx['txn_gateway_txn_id'])) {
                        $tag = $_tx['txn_gateway_txn_id'];
                    }
                    $fee = 0;
                    if (isset($_tx['txn_gateway_fee'])) {
                        $fee = (int) $_tx['txn_gateway_fee'];
                    }
                    $tax = 0;
                    if (isset($_tx['txn_tax'])) {
                        $tax = (int) $_tx['txn_tax'];
                    }
                    // We have to set a couple things up to get this recorded in the register
                    $_it = array(
                        '_item_id'      => 0,
                        '_profile_id'   => 0,
                        'cart_amount'   => $_tx['txn_total'],
                        'cart_module'   => 'jrPayment',
                        'cart_type'     => 'payment',
                        'cart_quantity' => 1
                    );
                    if (!jrPayment_record_sale_in_register($tid, $_tx['txn_id'], $_tx['txn_user_id'], $_it, $tag, $fee, $tax)) {
                        jrCore_logger('CRI', "Payments: unable to record charge in register", array('_tx' => $_tx, '_it' => $_it));
                    }

                }

                //------------------------
                // CART TRANSACTION
                //------------------------
                // If we come out of our event with a txn_cart_id, it means
                // we have a properly formatted CART PURCHASE transaction
                elseif (isset($_tx['txn_cart_id']) && jrCore_checktype($_tx['txn_cart_id'], 'number_nz')) {

                    // Get cart associated with this transaction
                    $_cr = jrPayment_get_cart_by_id($_tx['txn_cart_id']);
                    if (!$_cr || !is_array($_cr)) {
                        jrCore_logger('CRI', "Payments: cart not found for incoming transaction", $_tx);
                        jrPayment_run_plugin_function('webhook_response', 'success', $_post);
                        header('HTTP/1.0 200 OK');
                        exit;
                    }
                    $_tx['txn_user_id'] = $_cr['cart_user_id'];

                    // Validate that the cart has not been tampered with
                    if (!jrPayment_validate_cart($_cr, $_tx)) {
                        jrCore_logger('CRI', "Payments: cart validation hash mismatch", array('cart' => $_cr, 'transaction' => $_tx));
                        jrPayment_run_plugin_function('webhook_response', 'failure', $_post);
                        header('HTTP/1.0 400 Bad Response');
                        exit;
                    }
                    $_tx['txn_item_count'] = count($_cr['_items']);

                    // Update transaction
                    $_up = array(
                        'txn_user_id'    => $_tx['txn_user_id'],
                        'txn_item_count' => $_tx['txn_item_count']
                    );
                    if (isset($_cr['cart_charge'])) {
                        $_up['txn_charge'] = (int) $_cr['cart_charge'];
                    }
                    jrCore_db_update_item('jrPayment', $tid, $_up);

                    // Cart items?
                    if (isset($_cr['_items']) && count($_cr['_items']) > 0) {

                        // If this is an ACTIVE transaction, we can process it now.
                        // If pending, we have to save for later when payment comes through
                        if ($_tx['txn_status'] == 'active') {

                            // Count sold items
                            jrCore_create_stat_entry('jrPayment', 'item_count', 'cart', 0, 0, null, false, $_tx['txn_item_count']);

                            // Mark this cart as having completed checkout
                            jrPayment_set_cart_status($_tx['txn_cart_id'], PAYMENT_CART_COMPLETE);

                            // Trigger payment_cart_complete event
                            $_cr = jrCore_trigger_event('jrPayment', 'cart_checkout_complete', $_cr, $_tx);

                            // Get user info that made this purchase
                            $_us = jrCore_db_get_item('jrUser', $_cr['cart_user_id']);

                            // Record our register entry for each item sold
                            $_pi = array();
                            foreach ($_cr['_items'] as $k => $_i) {

                                $tag = null;
                                if (isset($_tx['txn_gateway_txn_id'])) {
                                    $tag = $_tx['txn_gateway_txn_id'];
                                }
                                $fee = 0;
                                if (isset($_tx['txn_gateway_fee'])) {
                                    $fee = (int) $_tx['txn_gateway_fee'];
                                }
                                $tax = 0;
                                if (isset($_tx['txn_tax'])) {
                                    $tax = (int) $_tx['txn_tax'];
                                }
                                if (!jrPayment_record_sale_in_register($tid, $_tx['txn_id'], $_tx['txn_user_id'], $_i, $tag, $fee, $tax)) {
                                    jrCore_logger('CRI', "Payments: unable to record sale in register", array('_tx' => $_tx, '_cr' => $_cr, '_i' => $_i));
                                    continue;
                                }

                                // Increment sale counter for this item
                                if ($pfx = jrCore_db_get_prefix($_i['cart_module'])) {
                                    jrCore_db_increment_key($_i['cart_module'], $_i['_item_id'], "{$pfx}_sale_count", 1);
                                }
                                // Increment Profile sale count
                                jrCore_db_increment_key('jrProfile', $_i['_profile_id'], 'profile_jrPayment_sale_count', 1);

                                // Let the purchasing module or other listeners process this sale
                                jrCore_trigger_event('jrPayment', 'purchase_item', $_i, $_cr);

                                $pid = (int) $_i['_profile_id'];
                                if (!isset($_pi[$pid])) {
                                    $_pi[$pid] = array(
                                        '_items' => array()
                                    );
                                }
                                $_cr['_items'][$k]['item_name'] = jrPayment_get_sold_item_name($_i);
                                $_pi[$pid]['_items'][]          = $_cr['_items'][$k];
                            }

                            // Notify profiles of sale
                            if (count($_pi) > 0) {
                                foreach ($_pi as $pid => $_items) {
                                    jrPayment_notify_profile_of_sold_items($pid, $_items, $_us);
                                }
                            }

                            // Send the buyer their receipt
                            jrPayment_send_buyer_receipt($_cr['cart_user_id'], $_cr['_items']);

                            // Notify site admins
                            $_cr['txn_id'] = $tid;
                            jrPayment_notify_admins_of_sold_items($_cr, $_us);

                        }
                        elseif ($_tx['txn_status'] == 'pending') {

                            // For a PENDING payment, we just mark the cart as COMPLETE -
                            // when the actual payment clears, the status will be ACTIVE
                            // and the cart will be processed at that time
                            // Mark this cart as having completed checkout
                            jrPayment_set_cart_status($_tx['txn_cart_id'], PAYMENT_CART_PENDING);

                            // Get user info that made this purchase
                            if ($_us = jrCore_db_get_item('jrUser', $_cr['cart_user_id'])) {

                                // Notify them that their purchase is pending
                                jrPayment_notify_buyer_of_pending_purchase($_us, $_cr);

                            }

                        }
                    }

                }
            }
        }
        else {
            jrCore_logger('CRI', "Payments: error saving transaction id {$_tx['txn_id']} to the datastore", $_tx);
        }
    }

    jrPayment_run_plugin_function('webhook_response', 'success', $_post);
    header('HTTP/1.0 200 OK');
    exit;
}
