<?php
 /**
 * Jamroom Redis Cache module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2003 - 2022 Talldude Networks, LLC.
 * @noinspection PhpExpressionResultUnusedInspection
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Delete all keys in Redis
 * @param string $module Optionally delete all cache entries for specific module
 * @param mixed $user_id int|array Optionally delete all cache entries for a specific user_id or array of user_id's
 * @return bool
 */
function _jrRedis_redis_delete_all_cache_entries($module = null, $user_id = null)
{
    global $_mods;
    if (!class_exists('Redis')) {
        return jrRedis_fallback('cache', 'delete_all_cache_entries', array($module, $user_id));
    }
    $_sv = jrRedis_get_configured_servers_by_type('cache');
    if ($_sv && is_array($_sv)) {

        $pfx = jrRedis_get_key_prefix();
        foreach ($_sv as $sid => $config) {
            if ($rds = jrRedis_connect('cache', $sid)) {

                // Deleting ALL cache entries for a site
                if (is_null($module) && is_null($user_id)) {
                    jrRedis_start_timer('maintenance');

                    // data structure:
                    // key                             = <encoding>:<expiration>:<value>
                    // hkey(<domain_key>m<dk>)         = key -> <module>,<pid>,<uid>
                    // hkey(<domain_key><profile_id>p) = key -> <module>
                    // hkey(<domain_key><user_id>u)    = key -> <module>
                    // hkey(<domain_key><module>p<dk>) = key -> <profile_id>,<created>
                    // hkey(<domain_key><module>u<dk>) = key -> <user_id>,<created>

                    // If we have Single Site enabled, flush DB
                    if (jrCore_get_config_value('jrRedis', 'single_site', 'off') == 'on') {
                        // We are the ONLY site - just flush the DB
                        $rds->flushDb();
                    }
                    else {
                        // We are sharing this Redis instance - delete key by key
                        $_dk = jrRedis_get_all_distribution_keys();
                        foreach ($_dk as $dky) {
                            foreach ($_mods as $mod => $_inf) {

                                // First - get the profile and user map keys - this gets us the list of the keys we have to remove
                                $_ky = $rds->multi(Redis::PIPELINE)->hGetAll("{$pfx}{$mod}p{$dky}")->hGetAll("{$pfx}{$mod}u{$dky}")->del("{$pfx}{$mod}p{$dky}")->del("{$pfx}{$mod}u{$dky}")->del("{$pfx}m{$dky}")->exec();
                                if (!empty($_ky[0])) {
                                    // This is PROFILE mapped keys
                                    $rds->multi(Redis::PIPELINE);
                                    foreach ($_ky[0] as $key => $line) {
                                        $rds->del($key);
                                        list($pid,) = explode(',', $line, 2);
                                        foreach (explode(':', $pid) as $p) {
                                            $rds->del("{$pfx}{$p}p");
                                        }
                                    }
                                    $rds->exec();
                                }

                                if ($_ky && isset($_ky[1]) && is_array($_ky[1]) && count($_ky[1]) > 0) {
                                    // This is USER mapped keys
                                    $rds->multi(Redis::PIPELINE);
                                    foreach ($_ky[1] as $key => $line) {
                                        list($uid,) = explode(',', $line, 2);
                                        $rds->del($key)->del("{$pfx}{$uid}u");
                                    }
                                    $rds->exec();
                                }
                            }
                        }
                    }
                    jrRedis_stop_timer('maintenance');
                }

                // Deleting all cache entries for a specific MODULE
                elseif (!is_null($module) && is_null($user_id)) {
                    if (isset($_mods[$module])) {
                        $_dk = jrRedis_get_all_distribution_keys();
                        jrRedis_start_timer('maintenance');
                        foreach ($_dk as $dky) {
                            // p = profile keys
                            // u = user keys
                            $_ky = $rds->multi(Redis::PIPELINE)->hGetAll("{$pfx}{$module}p{$dky}")->hGetAll("{$pfx}{$module}u{$dky}")->del("{$pfx}{$module}p{$dky}")->del("{$pfx}{$module}u{$dky}")->exec();

                            if ($_ky && isset($_ky[0]) && is_array($_ky[0])) {
                                $rds->multi(Redis::PIPELINE);
                                foreach ($_ky[0] as $key => $line) {
                                    $rds->del($key)->hDel("{$pfx}m{$dky}", $key);
                                    list($pid,) = explode(',', $line, 2);
                                    foreach (explode(':', $pid) as $p) {
                                        $rds->hDel("{$pfx}{$p}p", $key);
                                    }
                                }
                                $rds->exec();
                            }

                            if ($_ky && isset($_ky[1]) && is_array($_ky[1])) {
                                $rds->multi(Redis::PIPELINE);
                                foreach ($_ky[1] as $key => $line) {
                                    list($uid,) = explode(',', $line, 2);
                                    $rds->del($key)->hDel("{$pfx}m{$dky}", $key)->hDel("{$pfx}{$uid}u", $key);
                                }
                                $rds->exec();
                            }
                        }
                        jrRedis_stop_timer('maintenance');
                    }
                }

                // Deleting all cache entries for a single user_id OR an array of user_id's
                elseif (is_null($module) && !is_null($user_id)) {
                    if (is_array($user_id)) {
                        jrRedis_start_timer('maintenance');
                        $rds->multi(Redis::PIPELINE);
                        foreach ($user_id as $uid) {
                            $rds->hGetAll("{$pfx}{$uid}u")->del("{$pfx}{$uid}u");
                        }
                        $_ky = $rds->exec();
                        if ($_ky && is_array($_ky)) {
                            $rds->multi(Redis::PIPELINE);
                            foreach ($_ky as $idx => $_keys) {
                                if (($idx % 2) === 0) {
                                    foreach ($_keys as $key => $mod) {
                                        $dky = jrRedis_get_distribution_key($key);
                                        $rds->del($key)->hDel("{$pfx}m{$dky}", $key)->hDel("{$pfx}{$mod}u{$dky}", $key);
                                    }
                                }
                            }
                            $rds->exec();
                        }
                        jrRedis_stop_timer('maintenance');
                    }
                    else {
                        $uid = (int) $user_id;
                        if ($uid > 0) {
                            $_ky = $rds->multi(Redis::PIPELINE)->hGetAll("{$pfx}{$uid}u")->del("{$pfx}{$uid}u")->exec();
                            if ($_ky && isset($_ky[0]) && is_array($_ky[0])) {
                                jrRedis_start_timer('maintenance');
                                $rds->multi(Redis::PIPELINE);
                                foreach ($_ky[0] as $key => $mod) {
                                    $dky = jrRedis_get_distribution_key($key);
                                    $rds->del($key)->hDel("{$pfx}m{$dky}", $key)->hDel("{$pfx}{$mod}u{$dky}", $key);
                                }
                                $rds->exec();
                                jrRedis_stop_timer('maintenance');
                            }
                        }
                    }
                }

                // Deleting all cache entries for a specific MODULE for a single user_id OR array of user_id's
                else {
                    if (is_array($user_id)) {
                        jrRedis_start_timer('maintenance');
                        $rds->multi(Redis::PIPELINE);
                        $_map = array();
                        foreach ($user_id as $uid) {
                            $rds->hGetAll("{$pfx}{$uid}u");
                            $_map[] = $uid;
                        }
                        $_ky = $rds->exec();
                        if ($_ky && is_array($_ky)) {
                            $rds->multi(Redis::PIPELINE);
                            foreach ($_ky as $_keys) {
                                foreach ($_keys as $key => $mod) {
                                    $uid = $_map[$key];
                                    $dky = jrRedis_get_distribution_key($key);
                                    $rds->del($key)->hDel("{$pfx}{$uid}u", $key)->hDel("{$pfx}m{$dky}", $key)->hDel("{$pfx}{$mod}u{$dky}", $key);
                                }
                            }
                            $rds->exec();
                        }
                        jrRedis_stop_timer('maintenance');
                    }
                    else {
                        $uid = (int) $user_id;
                        if ($uid > 0) {
                            $_ky = $rds->hGetAll("{$pfx}{$uid}u");
                            if ($_ky && is_array($_ky)) {
                                $_dl = array_filter($_ky, function ($mod) use ($module) {
                                    return $mod == $module;
                                });
                                unset($_ky);
                                if (count($_dl) > 0) {
                                    jrRedis_start_timer('maintenance');
                                    $rds->multi(Redis::PIPELINE);
                                    foreach ($_dl as $key => $mod) {
                                        $dky = jrRedis_get_distribution_key($key);
                                        $rds->del($key)->hDel("{$pfx}{$uid}u", $key)->hDel("{$pfx}m{$dky}", $key)->hDel("{$pfx}{$mod}u{$dky}", $key);
                                    }
                                    $rds->exec();
                                    jrRedis_stop_timer('maintenance');
                                }
                            }
                        }
                    }
                }
            }
            else {
                return jrRedis_fallback('cache', 'delete_all_cache_entries', array($module, $user_id));
            }
        }
    }
    return true;
}

/**
 * Redis cache function to reset profile cache
 * @param $_profile_ids array Array of profile_id => module entries to delete
 * @return bool
 */
function _jrRedis_redis_process_exit_delete_profile_cache($_profile_ids)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('cache', 'process_exit_delete_profile_cache', array($_profile_ids));
    }
    $_sv = jrRedis_get_configured_servers_by_type('cache');
    if ($_sv && is_array($_sv)) {

        $pfx = jrRedis_get_key_prefix();
        foreach ($_sv as $sid => $config) {
            if ($rds = jrRedis_connect('cache', $sid)) {
                jrRedis_start_timer('access');

                // data structure:
                // key                               = <encoding>:<expiration>:<value>
                // hkey(<prefix>m<dist_key>)         = key -> <module>,<pid>,<uid>
                // hkey(<prefix><profile_id>p)       = key -> <module>
                // hkey(<prefix><user_id>u)          = key -> <module>
                // hkey(<prefix><module>p<dist_key>) = key -> <profile_id>,<created>
                // hkey(<prefix><module>u<dist_key>) = key -> <user_id>,<created>

                $rds->multi(Redis::PIPELINE);
                foreach ($_profile_ids as $pid => $module) {
                    $rds->echo("{$pid},{$module}")->hGetAll("{$pfx}{$pid}p");
                }
                $_ky = $rds->exec();
                if ($_ky && is_array($_ky)) {
                    $pid    = 0;
                    $fnd    = false;
                    $module = '';
                    $rds->multi(Redis::PIPELINE);
                    foreach ($_ky as $_set) {
                        if (!is_array($_set)) {
                            if (strpos($_set, ',')) {
                                // This is our PROFILE ID and MODULE
                                list($pid, $module) = explode(',', $_set);
                                $pid    = (int) $pid;
                                $module = trim($module);
                            }
                            continue;
                        }
                        // Fall through - we should have a $pid set now
                        if ($pid > 0) {
                            foreach ($_set as $key => $mod) {
                                if ($mod == $module) {
                                    // We are deleting specific module cache for this profile ID
                                    $dky = jrRedis_get_distribution_key($key);
                                    $rds->del($key)->hDel("{$pfx}{$pid}p", $key)->hDel("{$pfx}m{$dky}", $key)->hDel("{$pfx}{$mod}p{$dky}", $key);
                                    $fnd = true;
                                }
                                elseif ($module == 0) {
                                    // We are deleting ALL cache for this profile ID
                                    $dky = jrRedis_get_distribution_key($key);
                                    $rds->del($key)->del("{$pfx}{$pid}p")->hDel("{$pfx}m{$dky}", $key)->hDel("{$pfx}{$mod}p{$dky}", $key);
                                    $fnd = true;
                                }
                            }
                        }
                    }
                    if ($fnd) {
                        $rds->exec();
                    }
                    else {
                        $rds->discard();
                    }
                }

                jrRedis_stop_timer('access');
                jrRedis_disconnect($rds);
            }
        }
        return true;
    }
    return false;
}

/**
 * Redis cache function to delete skin cache for cleared profile_ids
 * @param $_profile_ids array Array of profile_id => module entries to delete
 * @return bool
 * @deprecated
 */
function _jrRedis_redis_process_exit_delete_profile_skin_cache($_profile_ids)
{
    // Deprecated - not used
    return true;
}

/**
 * Cache: Delete cache for a given key
 * @param string $module Module to save cache for
 * @param string $key Key to save cache for
 * @return bool
 */
function _jrRedis_redis_delete_cache($module, $key)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('cache', 'delete_cache', array($module, $key));
    }
    if ($rds = jrRedis_key_connect('cache', $key)) {
        $dky = jrRedis_get_distribution_key($key);
        $pfx = jrRedis_get_key_prefix();
        $key = jrRedis_get_packed_key($key);
        jrRedis_start_timer('access');
        $_ky = $rds->multi(Redis::PIPELINE)->hGet("{$pfx}m{$dky}", $key)->del($key)->exec();
        if ($_ky && is_array($_ky) && isset($_ky[0])) {
            list($mod, $pid, $uid) = explode(',', $_ky[0], 3);
            $rds->multi()->hDel("{$pfx}{$mod}p{$dky}", $key)->hDel("{$pfx}{$mod}u{$dky}", $key)->hDel("{$pfx}{$uid}u", $key);
            foreach (explode(':', $pid) as $p) {
                $rds->hDel("{$pfx}{$p}p", $key);
            }
            $rds->exec();
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        return true;
    }
    return jrRedis_fallback('cache', 'delete_cache', array($module, $key));
}

/**
 * Cache: Delete cache for multiple module/keys
 * @param array $_items Cache Items to delete
 * @return bool
 */
function _jrRedis_redis_delete_multiple_cache_entries($_items)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('cache', 'delete_multiple_cache_entries', array($_items));
    }
    if (is_array($_items)) {
        $_sv = jrRedis_get_configured_servers_by_type('cache');
        if ($_sv && is_array($_sv)) {

            $pfx = jrRedis_get_key_prefix();
            foreach ($_sv as $sid => $config) {

                if (count($_items) > 0) {
                    if ($rds = jrRedis_connect('cache', $sid)) {
                        jrRedis_start_timer('access');

                        $rds->multi(Redis::PIPELINE);
                        foreach ($_items as $v) {
                            // $v[0] = module
                            // $v[1] = key
                            $dky  = jrRedis_get_distribution_key($v[1]);
                            $v[1] = jrRedis_get_packed_key($v[1]);
                            $rds->hGet("{$pfx}m{$dky}", $v[1])->echo($v[1])->hDel("{$pfx}m{$dky}", $v[1])->hDel("{$pfx}{$v[0]}p{$dky}", $v[1])->hDel("{$pfx}{$v[0]}u{$dky}", $v[1])->del($v[1]);
                        }
                        $_ky = $rds->exec();

                        if ($_ky && is_array($_ky)) {
                            $rds->multi(Redis::PIPELINE);
                            $num = 0;
                            $_dk = array();
                            foreach ($_ky as $k => $v) {
                                if (strpos($v, ',')) {
                                    list(, $pid, $uid) = explode(',', $v, 3);
                                    $idx = ($k + 1);
                                    $key = $_ky[$idx];
                                    $rds->hDel("{$pfx}{$uid}u", $key);
                                    foreach (explode(':', $pid) as $p) {
                                        $rds->hDel("{$pfx}{$p}p", $key);
                                    }
                                    $_dk[$num] = $key;
                                    $num++;
                                    $_dk[$num] = $key;
                                    $num++;
                                }
                            }
                            $_rs = $rds->exec();
                            if ($_rs && isset($_rs[0])) {
                                // We deleted at least 1 key on this server - remove from $_items
                                foreach ($_rs as $k => $v) {
                                    if ($v == 1) {
                                        $key = $_dk[$k];
                                        if (isset($_items[$key])) {
                                            unset($_items[$key]);
                                        }
                                    }
                                }
                            }
                        }
                        jrRedis_stop_timer('access');
                        jrRedis_disconnect($rds);
                    }
                    else {
                        jrRedis_fallback('cache', 'delete_multiple_cache_entries', array($_items));
                    }
                }
            }
            return true;
        }
    }
    return false;
}

/**
 * Delete expired cache entries from the cache table
 * @note runs hourly in the core hourly maintenance listener
 */
function _jrRedis_redis_cache_maintenance()
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('cache', 'cache_maintenance', array());
    }
    // Redis will expire main keys on it's own - we have to expire module, profile and user keys
    @ini_set('memory_limit', '512M');
    $_sv = jrRedis_get_configured_servers_by_type('cache');
    if ($_sv && is_array($_sv)) {
        $pfx = jrRedis_get_key_prefix();
        foreach ($_sv as $sid => $config) {
            if ($rds = jrRedis_connect('cache', $sid)) {
                jrRedis_start_timer('maintenance');

                // Get all mapped keys
                $_ky = jrRedis_get_all_distribution_keys();
                foreach ($_ky as $key) {

                    $_mp = $rds->hGetAll("{$pfx}m{$key}");
                    if ($_mp && count($_mp) > 0) {
                        $rds->multi(Redis::PIPELINE);
                        foreach ($_mp as $k => $mpu) {
                            $rds->exists($k);
                        }
                        $_ex = $rds->exec();
                        $idx = 0;
                        $_dl = array();
                        foreach ($_mp as $k => $mpu) {
                            if (!isset($_ex[$idx]) || $_ex[$idx] == 0) {
                                // This key no longer exists
                                $_dl[$k] = $mpu;
                            }
                            $idx++;
                        }
                        unset($_ex);
                        if (count($_dl) > 0) {
                            $rds->multi(Redis::PIPELINE);
                            foreach ($_dl as $k => $mpu) {
                                list($mod, $pid, $uid) = explode(',', $mpu, 3);
                                $dky = jrRedis_get_distribution_key($k);
                                $rds->hDel("{$pfx}{$mod}p{$dky}", $k)->hDel("{$pfx}{$mod}u{$dky}", $k)->hDel("{$pfx}{$uid}u", $k)->hDel("{$pfx}m{$dky}", $k);
                                foreach (explode(':', $pid) as $p) {
                                    $rds->hDel("{$pfx}{$p}p", $k);
                                }
                            }
                            $rds->exec();
                        }
                    }
                }
                jrRedis_stop_timer('maintenance');
                jrRedis_disconnect($rds);
            }
        }
    }
    return true;
}

/**
 * Cache: Check if a given key is cached and return it's value
 * @param string $module Module to save cache for
 * @param string $key Key to save cache for
 * @param bool $add_user By default each cache entry is for a specific User ID - set to false to override
 * @return mixed returns string on success, bool false on not cached
 */
function _jrRedis_redis_is_cached($module, $key, $add_user = true)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('cache', 'is_cached', array($module, $key, $add_user));
    }
    if ($rds = jrRedis_key_connect('cache', $key)) {
        $key = jrRedis_get_packed_key($key);
        jrRedis_start_timer('access');
        try {
            $val = $rds->get($key);
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('read_error');
            return false;
        }
        jrRedis_stop_timer('access');
        if ($val) {
            if (strlen($val) > 1) {
                list($enc, $exp, $val) = explode(':', $val, 3);
                // Have we expired?
                $now = time();
                if ($exp <= $now) {
                    // We've expired - to prevent a rebuild stampede we:
                    // 1) Update the key with a NEW expiration of 30 seconds from now
                    // 2) return false on THIS request so the value is recreated
                    $exp = ($now + 30);
                    jrRedis_start_timer('access');
                    try {
                        $rds->setEx($key, 30, "{$enc}:{$exp}:{$val}");
                    }
                    catch (Exception $e) {
                        jrRedis_stop_timer('access');
                        jrRedis_record_event('write_error');
                        jrRedis_disconnect($rds);
                        return false;
                    }
                    jrRedis_stop_timer('access');
                    jrRedis_disconnect($rds);
                    return false;
                }
                switch ($enc) {
                    case 1:  // array
                        $val = jrRedis_string_to_array($val);
                        break;
                    case 2:  // object
                        $val = jrRedis_string_to_array($val, false);
                        break;
                }
                jrRedis_stop_timer('access');
                jrRedis_disconnect($rds);
                return $val;
            }
        }
        jrRedis_disconnect($rds);
        return false;
    }
    return jrRedis_fallback('cache', 'is_cached', array($module, $key, $add_user));
}

/**
 * Cache: Cache a string for a given key
 * @param string $module Module doing the caching
 * @param string $key Unique key for cache item
 * @param mixed $value Value to cache
 * @param int $expire_seconds How long key will be cached for (in seconds)
 * @param int $profile_id Profile ID cache item belongs to - 0 = "system"
 * @param bool $add_user By default each cache entry is for a specific User ID - set to false to override
 * @param string $unique comma separated list of Profile IDs (set in datastore)
 * @return mixed returns string on success, bool false on not cached
 */
function _jrRedis_redis_add_to_cache($module, $key, $value, $expire_seconds = 0, $profile_id = 0, $add_user = true, $unique = null)
{
    global $_post, $_conf;
    if (!class_exists('Redis')) {
        return jrRedis_fallback('cache', 'add_to_cache', array($module, $key, $value, $expire_seconds, $profile_id, $add_user, $unique));
    }

    // @see https://github.com/phpredis/phpredis

    // data structure:
    // Contains the actual key data, encoding and expiration
    // key                             = <encoding>:<expiration>:<value>

    // This is the main "map" key - there are 16 of these hashes
    // that keep track of the key, profile_id and user_id
    // hkey(<domain_key>m<dk>)         = key -> <module>,<pid>,<uid>

    // This profile_id hash keeps track of which keys belong to which module
    // hkey(<domain_key><profile_id>p) = key -> <module>

    // The user_id hash keeps track of which key belongs to which module
    // hkey(<domain_key><user_id>u)    = key -> <module>

    // The module profile_id hash keeps track of key to profile_id mappings
    // hkey(<domain_key><module>p<dk>) = key -> <profile_id>,<created>

    // The module user_id hash keeps track of key to profile_id mappings
    // hkey(<domain_key><module>u<dk>) = key -> <user_id>,<created>

    if (empty($expire_seconds)) {
        $expire_seconds = $_conf['jrCore_default_cache_seconds'];
    }
    $expire_seconds = intval($expire_seconds);
    if ($expire_seconds === 0) {
        return true;
    }

    // Check if we are encoding this in the DB
    $enc = 0;
    if (is_array($value)) {
        $val = jrRedis_array_to_string($value);
        $enc = 1;
    }
    elseif (is_object($value)) {
        $val = jrRedis_array_to_string($value);
        $enc = 2;
    }
    else {
        $val = $value;
    }

    if ($profile_id === 0 && isset($_post['_profile_id'])) {
        $pid = (int) $_post['_profile_id'];
    }
    else {
        $pid = (int) $profile_id;
    }

    $uid = (isset($_SESSION['_user_id'])) ? (int) $_SESSION['_user_id'] : 0;
    $now = time();

    if ($rds = jrRedis_key_connect('cache', $key)) {
        $pfx = jrRedis_get_key_prefix();
        $exp = ($now + $expire_seconds);
        $dky = jrRedis_get_distribution_key($key);
        $key = jrRedis_get_packed_key($key);
        jrRedis_start_timer('access');
        $_pd = array($pid => $pid);
        if (!is_null($unique)) {
            foreach (explode(',', $unique) as $profile_id) {
                $profile_id       = intval($profile_id);
                $_pd[$profile_id] = $profile_id;
            }
            $pid = implode(':', $_pd);
        }
        try {
            $rds->multi(Redis::PIPELINE)->
            setEx($key, $expire_seconds, "{$enc}:{$exp}:{$val}")->
            hSet("{$pfx}m{$dky}", $key, "{$module},{$pid},{$uid}")->
            hSet("{$pfx}{$uid}u", $key, $module)->
            hSet("{$pfx}{$module}p{$dky}", $key, "{$pid},{$now}")->
            hSet("{$pfx}{$module}u{$dky}", $key, "{$uid},{$now}");
            foreach ($_pd as $p) {
                $rds->hSet("{$pfx}{$p}p", $key, $module);
            }
            $rds->exec();
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return false;
        }

        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        return true;
    }
    return jrRedis_fallback('cache', 'add_to_cache', array($module, $key, $value, $expire_seconds, $profile_id, $add_user, $unique));
}
