<?php
 /**
 * Jamroom Redis Cache module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2003 - 2022 Talldude Networks, LLC.
 * @noinspection PhpExpressionResultUnusedInspection
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Get number of seconds until a form session auto-expires
 * @return int
 */
function jrRedis_get_form_expire_seconds()
{
    return (jrCore_get_config_value('jrCore', 'form_session_expire_hours', 4) * 3600);
}

/**
 * Create a new Form Session
 * @param string $form_id Form ID from jrCore_form_token_create()
 * @param array $_form Form Information
 * @return mixed
 */
function _jrRedis_redis_form_create_session($form_id, $_form)
{
    global $_post;
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_create_session', array($form_id, $_form));
    }

    $opt = "{$_post['module']}/{$_post['option']}";
    $key = jrRedis_get_distribution_key($form_id);
    $exp = jrRedis_get_form_expire_seconds();
    $pfx = jrRedis_get_key_prefix();

    // Update an Existing Form Session
    if ($_rt = _jrRedis_redis_form_get_session($form_id)) {
        // We do - update some info and return
        $_rt['form_updated'] = time();
        $_rt['form_rand']    = mt_rand(0, 999999999);
        $_rt['form_view']    = $opt;
        $_rt['form_params']  = $_form;
        if ($rds = jrRedis_key_connect('form_session', $form_id)) {
            $form_data = jrRedis_array_to_string($_rt);
            jrRedis_start_timer('access');
            try {
                $rds->multi()->setEx("{$pfx}{$form_id}", $exp, $form_data)->hSet("{$pfx}fs{$key}", $form_id, "{$opt}:" . time())->exec();
            }
            catch (Exception $e) {
                jrRedis_stop_timer('access');
                jrRedis_record_event('write_error');
                jrRedis_disconnect($rds);
                return jrRedis_fallback('form_session', 'form_create_session', array($form_id, $_form));
            }
            jrRedis_stop_timer('access');
            jrRedis_disconnect($rds);
            return $_rt;
        }
        return jrRedis_fallback('form_session', 'form_create_session', array($form_id, $_form));
    }

    // Create a new form session
    if ($rds = jrRedis_key_connect('form_session', $form_id)) {

        // Create new form session
        $uid       = (isset($_SESSION['_user_id'])) ? intval($_SESSION['_user_id']) : 0;
        $_dt       = array(
            'form_token'     => $form_id,
            'form_created'   => time(),
            'form_updated'   => 0,
            'form_rand'      => 0,
            'form_user_id'   => $uid,
            'form_view'      => $opt,
            'form_validated' => 0,
            'form_params'    => $_form
        );
        $form_data = jrRedis_array_to_string($_dt);
        jrRedis_start_timer('access');
        try {
            $rds->multi()->setEx("{$pfx}{$form_id}", $exp, $form_data)->hSet("{$pfx}fs{$key}", $form_id, "{$opt}:" . time())->exec();
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return jrRedis_fallback('form_session', 'form_create_session', array($form_id, $_form));
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        $_dt['form_fields'] = '';
        $_dt['form_saved']  = '';
        return $_dt;
    }
    return jrRedis_fallback('form_session', 'form_create_session', array($form_id, $_form));
}

/**
 * Get information about the active form session
 * @param string $form_id Form ID to get session for
 * @return mixed
 */
function _jrRedis_redis_form_get_session($form_id = null)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_get_session', array($form_id));
    }
    if ($rds = jrRedis_key_connect('form_session', $form_id)) {
        $pfx = jrRedis_get_key_prefix();
        jrRedis_start_timer('access');
        try {
            $val = $rds->get("{$pfx}{$form_id}");
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('read_error');
            jrRedis_disconnect($rds);
            return jrRedis_fallback('form_session', 'form_get_session', array($form_id));
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        if ($val) {
            return jrRedis_string_to_array($val);
        }
        return false;
    }
    return jrRedis_fallback('form_session', 'form_get_session', array($form_id));
}

/**
 * Save a form session
 * @param string $form_id Form ID of existing form session
 * @param array $_data Form information to save
 * @param array $_form existing form information
 * @return bool
 */
function _jrRedis_redis_form_save_session($form_id, $_data, $_form)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_save_session', array($form_id, $_data, $_form));
    }
    if ($rds = jrRedis_key_connect('form_session', $form_id)) {
        $exp                   = jrRedis_get_form_expire_seconds();
        $pfx                   = jrRedis_get_key_prefix();
        $_form['form_updated'] = time();
        $_form['form_saved']   = $_data;
        $form_data             = jrRedis_array_to_string($_form);
        jrRedis_start_timer('access');
        try {
            $rds->setEx("{$pfx}{$form_id}", $exp, $form_data);
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return jrRedis_fallback('form_session', 'form_save_session', array($form_id, $_data, $_form));
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        return true;
    }
    return jrRedis_fallback('form_session', 'form_save_session', array($form_id, $_data, $_form));
}

/**
 * Update a field in a form session
 * @param string $form_id Form ID of existing form session
 * @param array $_fields form field data
 * @return bool
 */
function _jrRedis_redis_form_update_session_fields($form_id, $_fields)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_update_session_fields', array($form_id, $_fields));
    }
    if ($rds = jrRedis_key_connect('form_session', $form_id)) {
        if ($_rt = _jrRedis_redis_form_get_session($form_id)) {
            $exp                = jrRedis_get_form_expire_seconds();
            $pfx                = jrRedis_get_key_prefix();
            $_rt['form_fields'] = $_fields;
            $form_data          = jrRedis_array_to_string($_rt);
            jrRedis_start_timer('access');
            try {
                $rds->setEx("{$pfx}{$form_id}", $exp, $form_data);
            }
            catch (Exception $e) {
                jrRedis_stop_timer('access');
                jrRedis_record_event('write_error');
                jrRedis_disconnect($rds);
                return jrRedis_fallback('form_session', 'form_update_session_fields', array($form_id, $_fields));
            }
            jrRedis_stop_timer('access');
            jrRedis_disconnect($rds);
            return true;
        }
        return false;
    }
    return jrRedis_fallback('form_session', 'form_update_session_fields', array($form_id, $_fields));
}

/**
 * Mark a form session as validated
 * @param string $form_id Form ID to delete
 * @return bool
 */
function _jrRedis_redis_form_mark_as_validated($form_id)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_mark_as_validated', array($form_id));
    }
    if ($rds = jrRedis_key_connect('form_session', $form_id)) {
        if ($_rt = _jrRedis_redis_form_get_session($form_id)) {
            $exp                   = jrRedis_get_form_expire_seconds();
            $pfx                   = jrRedis_get_key_prefix();
            $_rt['form_validated'] = 1;
            $form_data             = jrRedis_array_to_string($_rt);
            jrRedis_start_timer('access');
            try {
                $rds->setEx("{$pfx}{$form_id}", $exp, $form_data);
            }
            catch (Exception $e) {
                jrRedis_stop_timer('access');
                jrRedis_record_event('write_error');
                jrRedis_disconnect($rds);
                return jrRedis_fallback('form_session', 'form_mark_as_validated', array($form_id));
            }
            jrRedis_stop_timer('access');
            jrRedis_disconnect($rds);
            return true;
        }
        return false;
    }
    return jrRedis_fallback('form_session', 'form_mark_as_validated', array($form_id));
}

/**
 * Delete an Active form session
 * @param string $form_id Form ID to delete
 * @return bool
 */
function _jrRedis_redis_form_delete_session($form_id)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_delete_session', array($form_id));
    }
    if ($rds = jrRedis_key_connect('form_session', $form_id)) {
        if (_jrRedis_redis_form_get_session($form_id)) {
            $key = jrRedis_get_distribution_key($form_id);
            $pfx = jrRedis_get_key_prefix();
            jrRedis_start_timer('access');
            try {
                $rds->multi(Redis::PIPELINE)->del("{$pfx}{$form_id}")->hDel("{$pfx}fs{$key}", $form_id)->exec();
            }
            catch (Exception $e) {
                jrRedis_record_event('write_error');
            }
            jrRedis_stop_timer('access');
            jrRedis_disconnect($rds);
        }
        return true;
    }
    return jrRedis_fallback('form_session', 'form_delete_session', array($form_id));
}

/**
 * Perform form session maintenance
 * @param int $hours hours "old" to consider for session deletion
 * @return bool
 */
function _jrRedis_redis_form_session_maintenance($hours)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_session_maintenance', array($hours));
    }
    $_sv = jrRedis_get_configured_servers_by_type('form_session');
    if ($_sv && is_array($_sv)) {
        $_ky = jrRedis_get_all_distribution_keys();
        $pfx = jrRedis_get_key_prefix();
        foreach ($_sv as $server_id => $config) {
            jrRedis_start_timer('maintenance');
            if ($rds = jrRedis_connect('session', $server_id)) {
                $rds->multi(Redis::PIPELINE);
                foreach ($_ky as $key) {
                    $rds->hGetAll("{$pfx}fs{$key}");
                }
                $_ex = $rds->exec();
                if ($_ex && is_array($_ex)) {
                    $old = (time() - (intval($hours) * 3600));
                    $_dl = array();
                    foreach ($_ex as $_ids) {
                        foreach ($_ids as $form_id => $opt) {
                            list(, $time) = explode(':', $opt, 2);
                            if ($time < $old) {
                                $_dl[] = $form_id;
                            }
                        }
                    }
                    if (count($_dl) > 0) {
                        $rds->multi(Redis::PIPELINE);
                        foreach ($_dl as $form_id) {
                            $key = jrRedis_get_distribution_key($form_id);
                            $rds->del("{$pfx}{$form_id}")->hDel("{$pfx}fs{$key}", $form_id);
                        }
                        try {
                            $rds->exec();
                        }
                        catch (Exception $e) {
                            jrRedis_record_event('write_error');
                        }
                    }
                }
                jrRedis_stop_timer('maintenance');
                jrRedis_disconnect($rds);
            }
        }
    }
    return true;
}

/**
 * Delete all form sessions for a module/view
 * @param string $module Module that contains the View
 * @param string $view View to delete
 * @return bool
 */
function _jrRedis_Redis_form_delete_session_view($module, $view)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_delete_session_view', array($module, $view));
    }
    $_sv = jrRedis_get_configured_servers_by_type('form_session');
    if ($_sv && is_array($_sv)) {
        $_ky = jrRedis_get_all_distribution_keys();
        $pfx = jrRedis_get_key_prefix();
        foreach ($_sv as $server_id => $config) {
            jrRedis_start_timer('maintenance');
            if ($rds = jrRedis_connect('session', $server_id)) {
                $rds->multi(Redis::PIPELINE);
                foreach ($_ky as $key) {
                    $rds->hGetAll("{$pfx}fs{$key}");
                }
                $_ex = $rds->exec();
                if ($_ex && is_array($_ex)) {
                    $_dl = array();
                    foreach ($_ex as $_ids) {
                        foreach ($_ids as $form_id => $opt) {
                            list($opt,) = explode(':', $opt, 2);
                            if ($opt == "{$module}/{$view}") {
                                $_dl[] = $form_id;
                            }
                        }
                    }
                    if (count($_dl) > 0) {
                        $rds->multi(Redis::PIPELINE);
                        foreach ($_dl as $form_id) {
                            $key = jrRedis_get_distribution_key($form_id);
                            $rds->del("{$pfx}{$form_id}")->hDel("{$pfx}fs{$key}", $form_id);
                        }
                        $rds->exec();
                    }
                }
                jrRedis_stop_timer('maintenance');
                jrRedis_disconnect($rds);
            }
        }
    }
    return true;
}

/**
 * Delete all existing form sessions
 * @return mixed
 */
function _jrRedis_redis_form_delete_all_form_sessions()
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_delete_all_form_sessions', array());
    }
    // We're emptying ALL sessions
    $_sv = jrRedis_get_configured_servers_by_type('session');
    if ($_sv && is_array($_sv)) {
        $pfx = jrRedis_get_key_prefix();
        $_ky = jrRedis_get_all_distribution_keys();
        foreach ($_sv as $sid => $config) {
            if ($rds = jrRedis_connect('form_session', $sid)) {
                jrRedis_start_timer('maintenance');
                // If we have Single Site enabled, flush DB
                if (jrCore_get_config_value('jrRedis', 'single_site', 'off') == 'on') {
                    // We are the ONLY site - just flush the DB
                    $rds->flushDb();
                }
                else {
                    $rds->multi(Redis::PIPELINE);
                    foreach ($_ky as $key) {
                        $rds->hGetAll("{$pfx}fs{$key}");
                    }
                    $_ex = $rds->exec();
                    if ($_ex && is_array($_ex)) {
                        $_dl = array();
                        foreach ($_ex as $_ids) {
                            foreach ($_ids as $form_id => $opt) {
                                $_dl[] = $form_id;
                            }
                        }
                        if (count($_dl) > 0) {
                            $rds->multi(Redis::PIPELINE);
                            foreach ($_dl as $form_id) {
                                $key = jrRedis_get_distribution_key($form_id);
                                $rds->del("{$pfx}{$form_id}")->hDel("{$pfx}fs{$key}", $form_id);
                            }
                            $rds->exec();
                        }
                    }
                }
                jrRedis_stop_timer('maintenance');
                jrRedis_disconnect($rds);
            }
        }
        return true;
    }
    return false;
}
