<?php
 /**
 * Jamroom Redis Cache module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2003 - 2022 Talldude Networks, LLC.
 * @noinspection PhpExpressionResultUnusedInspection
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Set an error/notice message in a Modal Popup window
 * @param string $token Form Token (jr_html_modal_token)
 * @param string $type Type of Notice (error,notice,success,warning)
 * @param string $text Text for Notice
 * @return bool
 */
function _jrRedis_redis_form_modal_notice($token, $type, $text)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_modal_notice', array($token, $type, $text));
    }
    if ($rds = jrRedis_key_connect('form_session', $token)) {
        $pfx = jrRedis_get_key_prefix();
        jrRedis_start_timer('access');
        try {
            $rds->hSet("{$pfx}fm{$token}", microtime(true), "{$type}:{$text}");
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return false;
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        return true;
    }
    return false;
}

/**
 * Get modal entries for a modal token
 * @param $token
 * @return array|bool|mixed
 */
function _jrRedis_redis_form_modal_get_entries($token)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_modal_get_entries', array($token));
    }
    if ($rds = jrRedis_key_connect('form_session', $token)) {
        $pfx = jrRedis_get_key_prefix();
        jrRedis_start_timer('access');
        try {
            $_entries = $rds->multi(Redis::PIPELINE)->hGetAll("{$pfx}fm{$token}")->del("{$pfx}fm{$token}")->exec();
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('read_error');
            jrRedis_disconnect($rds);
            return false;
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        if (isset($_entries[0]) && is_array($_entries[0])) {
            ksort($_entries[0], SORT_NUMERIC);
            $_rt = array();
            foreach ($_entries[0] as $value) {
                list($t, $m) = explode(':', $value, 2);
                $_rt[] = array('t' => $t, 'm' => $m);
            }
            return $_rt;
        }
    }
    return false;
}

/**
 * Cleanup temp entries at the end of a Modal Window session
 * @param string $token Form Token (jr_html_modal_token)
 * @return bool
 */
function _jrRedis_redis_form_modal_cleanup($token)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('form_session', 'form_modal_cleanup', array($token));
    }
    if ($rds = jrRedis_key_connect('form_session', $token)) {
        $pfx = jrRedis_get_key_prefix();
        jrRedis_start_timer('access');
        try {
            $rds->del("{$pfx}fm{$token}");
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return false;
        }
        return true;
    }
    return false;
}

/**
 * Cleanup old for modal entries
 * @return bool
 */
function _jrRedis_redis_form_modal_maintenance()
{
    // Not needed with Redis
    return true;
}
