<?php
 /**
 * Jamroom Item Tags module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2013 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrTags_meta()
{
    return array(
        'name'        => 'Item Tags',
        'url'         => 'tags',
        'version'     => '2.3.1',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Add Tags to items to use in Search, listings and Tag Clouds',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/95/item-tags',
        'category'    => 'item features',
        'priority'    => 100,
        'requires'    => 'jrCore:6.5.12',
        'license'     => 'jcl'
    );
}

/**
 * init
 */
function jrTags_init()
{
    // Core Quota support
    $_options = array(
        'label' => 'Allowed to Add Tags',
        'help'  => 'If checked, users in this quota will be able to add tags to items that have an add tag form.'
    );
    jrCore_register_module_feature('jrCore', 'quota_support', 'jrTags', 'on', $_options);

    jrCore_register_module_feature('jrCore', 'javascript', 'jrTags', 'jrTags.js');
    jrCore_register_module_feature('jrCore', 'css', 'jrTags', 'jrTags.css');

    jrCore_register_module_feature('jrCore', 'javascript', 'jrTags', APP_DIR . '/modules/jrTags/contrib/jqcloud/jqcloud.js');
    jrCore_register_module_feature('jrCore', 'css', 'jrTags', APP_DIR . '/modules/jrTags/contrib/jqcloud/jqcloud.css');

    // Add additional search params
    jrCore_register_event_listener('jrCore', 'db_search_params', 'jrTags_db_search_params_listener');
    jrCore_register_event_listener('jrCore', 'db_create_item', 'jrTags_db_create_item_listener');
    jrCore_register_event_listener('jrCore', 'db_update_item', 'jrTags_db_update_item_listener');
    jrCore_register_event_listener('jrCore', 'verify_module', 'jrTags_verify_module_listener');

    // We offer a module detail feature for Item Tags
    $_tmp = array(
        'function' => 'jrTags_item_tags_feature',
        'label'    => 'Item Tags',
        'help'     => 'Adds a Tag listing and new tag entry form to Item Detail pages'
    );
    jrCore_register_module_feature('jrCore', 'item_detail_feature', 'jrTags', 'item_tags', $_tmp);

    // Site Builder widget
    jrCore_register_module_feature('jrSiteBuilder', 'widget', 'jrTags', 'widget_tag_cloud', 'Tag Cloud');

    // form field
    jrCore_register_module_feature('jrCore', 'form_field', 'jrTags', 'tags');

    // Verify Tags
    jrCore_register_queue_worker('jrTags', 'verify_tags', 'jrTags_verify_tags_worker', 0);

    // Recycle Bin support
    jrCore_register_module_feature('jrCore', 'recycle_bin_item_id_table', 'jrTags', 'tag_search', 'tag_module,tag_item_id');
    jrCore_register_module_feature('jrCore', 'recycle_bin_profile_id_table', 'jrTags', 'tag_search', 'tag_profile_id');

    return true;
}

//------------------------------------
// Queue Worker
//------------------------------------

/**
 * Convert tags DS to new structure
 * @param $_queue array Queue entry
 * @return bool
 */
function jrTags_verify_tags_worker($_queue)
{
    if (jrCore_db_get_datastore_item_count('jrTags') > 0) {
        $last_id = 0;
        $total   = 0;
        while (true) {
            // We need to convert our existing DS items to the new DB format
            $_sp = array(
                'search'         => array(
                    "_item_id > {$last_id}"
                ),
                'return_keys'    => array('tag_item_id', 'tag_profile_id', 'tag_module', 'tag_text'),
                'order_by'       => array('_item_id' => 'asc'),
                'skip_triggers'  => true,
                'privacy_check'  => false,
                'ignore_pending' => true,
                'limit'          => 1000
            );
            $_sp = jrCore_db_search_items('jrTags', $_sp);
            if ($_sp && is_array($_sp) && isset($_sp['_items'])) {
                if ($last_id == 0) {
                    jrCore_logger('INF', "migration of tags database to new format beginning");
                }
                $_in = array();
                foreach ($_sp['_items'] as $t) {
                    $mod     = jrCore_db_escape($t['tag_module']);
                    $pid     = (int) $t['tag_profile_id'];
                    $iid     = (int) $t['tag_item_id'];
                    $val     = jrCore_db_escape(jrTags_get_clean_tag(rawurldecode($t['tag_text'])));
                    $_in[]   = "('{$mod}',{$pid},{$iid},'{$val}')";
                    $last_id = (int) $t['_item_id'];
                    $total++;
                }
                if (count($_in)) {
                    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
                    $ins = "INSERT IGNORE INTO {$tbl} (tag_module, tag_profile_id, tag_item_id, tag_value) VALUES " . implode(',', $_in);
                    jrCore_db_query($ins);
                }
            }
            else {
                if ($total > 0) {
                    jrCore_logger('INF', "successfully migrated " . jrCore_number_format($total) . " tag entries to new database format");
                    jrCore_db_truncate_datastore('jrTags');
                }
                break;
            }
        }
    }
    return true;
}

//------------------------------------
// Widget
//------------------------------------

/**
 * Display Widget Config screen
 * @param $_post array Post info
 * @param $_user array User array
 * @param $_conf array Global Config
 * @param $_wg array Widget info
 * @return bool
 */
function jrTags_widget_tag_cloud_config($_post, $_user, $_conf, $_wg)
{
    // Tags Limit
    $_tmp = array(
        'name'     => 'limit',
        'label'    => 'Tag Limit',
        'help'     => 'What is the maximum number of tags that should appear in the Tag Cloud?',
        'default'  => 20,
        'type'     => 'text',
        'validate' => 'number_nz',
        'min'      => 1,
        'max'      => 100
    );
    jrCore_form_field_create($_tmp);
    return true;
}

/**
 * Get Widget results from posted Config data
 * @param $_post array Post info
 * @return array
 */
function jrTags_widget_tag_cloud_config_save($_post)
{
    return array('limit' => $_post['limit']);
}

/**
 * Display Widget
 * @param $_widget array Page Widget info
 * @return string
 */
function jrTags_widget_tag_cloud_display($_widget)
{
    $smarty = new stdClass;
    $params = array('limit' => $_widget['limit']);
    return smarty_function_jrTags_cloud($params, $smarty);
}

//------------------------------------
// Item Feature
//------------------------------------

/**
 * Return Existing Item tags and tag form
 * @param string $module Module item belongs to
 * @param array $_item Item info (from DS)
 * @param array $params Smarty function parameters
 * @param array $smarty current Smarty object
 * @return string
 */
function jrTags_item_tags_feature($module, $_item, $params, $smarty)
{
    global $_user;

    // See if we are enabled in this quota
    if (isset($_item['quota_jrTags_show_detail']) && $_item['quota_jrTags_show_detail'] == 'off') {
        return '';
    }
    // Is user only allow to tag this item?
    $can_tag = false;
    if (jrUser_is_admin()) {
        $can_tag = true;
    }
    elseif (isset($_user['quota_jrTags_allowed']) && $_user['quota_jrTags_allowed'] == 'on') {
        $can_tag = true;
        if (isset($_user['quota_jrTags_own_items_only']) && $_user['quota_jrTags_own_items_only'] == 'on' && !jrUser_can_edit_item($_item)) {
            $can_tag = false;
        }
    }
    if (!isset($params['class']) || strlen($params['class']) === 0) {
        $params['class'] = 'form_text';
    }
    if (!isset($params['style']) || strlen($params['style']) === 0) {
        $params['style'] = 'width:160px;';
    }
    $_tmp = array(
        'jrTags' => array(
            'module'     => $module,
            'profile_id' => $_item['_profile_id'],
            'item_id'    => $_item['_item_id'],
            'can_tag'    => $can_tag
        )
    );
    foreach ($params as $k => $v) {
        $_tmp['jrTags'][$k] = $v;
    }
    // Call the appropriate template and return
    return jrCore_parse_template('tag_form.tpl', $_tmp, 'jrTags');
}

//------------------------------------
// Smarty
//------------------------------------

/**
 * Smarty function to show an embedded tag adding input box.
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrTags_add($params, $smarty)
{
    global $_user;

    // Is jrTags module enabled?
    if (!jrCore_module_is_active('jrTags')) {
        return '';
    }
    // Is it allowed in this quota?
    if (!jrProfile_is_allowed_by_quota('jrTags', $smarty)) {
        return '';
    }
    // Check the incoming parameters
    if ($params['module'] == 'jrProfile') {
        $params['profile_id'] = $params['item_id'];
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return 'jrTags_form: Invalid profile_id';
    }
    if (!jrCore_checktype($params['item_id'], 'number_nz')) {
        return 'jrTags_form: Invalid item_id';
    }
    if (!jrCore_module_is_active($params['module'])) {
        return 'jrTags_form: Invalid or disabled module';
    }
    // Is user only allow to tag this item?
    $_item   = jrCore_db_get_item($params['module'], $params['item_id']);
    $can_tag = false;
    if (isset($_user['quota_jrTags_allowed']) && $_user['quota_jrTags_allowed'] == 'on') {
        $can_tag = true;
        if (isset($_user['quota_jrTags_own_items_only']) && $_user['quota_jrTags_own_items_only'] == 'on' && !jrUser_can_edit_item($_item)) {
            $can_tag = false;
        }
    }
    if (!isset($params['class']) || strlen($params['class']) === 0) {
        $params['class'] = 'form_text';
    }
    if (!isset($params['style']) || strlen($params['style']) === 0) {
        $params['style'] = 'width:160px;';
    }
    $_replace = array(
        'jrTags' => array(
            'can_tag' => $can_tag,
            'item'    => $_item
        )
    );
    foreach ($params as $k => $v) {
        $_replace['jrTags'][$k] = $v;
    }
    // Call the appropriate template and return
    $out = jrCore_parse_template('tag_form.tpl', $_replace, 'jrTags');
    if (isset($params['assign']) && $params['assign'] != '') {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Tag cloud
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrTags_cloud($params, $smarty)
{
    if (jrCore_get_config_value('jrTags', 'enable_cloud', 'on') == 'off') {
        // Not enabled
        return '';
    }
    // See if we are a mobile/table device with tag cloud turned off
    if (jrCore_is_mobile_device() && !jrCore_is_tablet_device()) {
        if (jrCore_get_config_value('jrTags', 'enable_mobile_cloud', 'off') == 'off') {
            // Not enabled for mobile phones
            return '';
        }
    }
    $_rp = array(
        'murl' => (isset($params['base_url'])) ? $params['base_url'] : jrCore_get_module_url('jrTags')
    );

    // Was a search passed in for a specific module?
    $search_module = null;
    if (!empty($params['search'])) {
        if (strpos(' ' . $params['search'], 'tag_module')) {
            // search="tag_module = `$tag_module`
            list(, , $search_module) = explode(' ', $params['search'], 3);
            if (!jrTags_is_disabled_module($search_module)) {
                $search_module = trim($search_module);
            }
        }
    }

    // got a profile id?
    if (isset($params['profile_id']) && jrCore_checktype($params['profile_id'], 'number_nz')) {

        // Is it allowed in this quota?
        if (!jrProfile_is_allowed_by_quota('jrTags', $smarty)) {
            return '';
        }
        $_rp['profile_url'] = jrCore_db_get_item_key('jrProfile', $params['profile_id'], 'profile_url');

        // Get all tags for this profile
        if (!is_null($search_module)) {
            $_rt = jrTags_get_all_module_tags_for_profile_id($search_module, $params['profile_id']);
        }
        else {
            $_rt = jrTags_get_all_tags_for_profile_id($params['profile_id']);
        }
    }
    else {

        $lim = 1000;
        if (isset($params['limit']) && jrCore_checktype($params['limit'], 'number_nz')) {
            $lim = (int) $params['limit'];
        }
        if (!is_null($search_module)) {
            $_rt = jrTags_get_all_module_tags($search_module, $lim);
        }
        else {
            $_rt = jrTags_get_all_tags($lim);
        }
    }

    $out = '';

    // weight them:
    $_sort = array();
    if ($_rt && is_array($_rt)) {
        foreach ($_rt as $tag) {
            if (strpos(' ' . $tag, '%')) {
                $tag = rawurldecode($tag);
            }
            if (!isset($_sort[$tag])) {
                $_sort[$tag] = 0;
            }
            $_sort[$tag]++;
        }
        arsort($_sort, SORT_NUMERIC);

        // Check for max number of tags to display in tag cloud
        $_rp['tags'] = array();
        $max         = (isset($params['max_tags']) && jrCore_checktype($params['max_tags'], 'number_nz')) ? intval($params['max_tags']) : 25;
        $_sort       = array_slice($_sort, 0, $max);
        $cnt         = ($max * 10);
        foreach ($_sort as $tag => $num) {
            if (!isset($_rp['tags'][$tag])) {
                $_rp['tags'][$tag] = array(
                    'tag_url'  => rawurlencode($tag),
                    'tag_text' => $tag,
                    'weight'   => ($cnt + (10 * (10 * $num)))
                );
                $cnt               -= 10;
            }
        }
        unset($_sort);

        foreach ($params as $k => $v) {
            $_rp['params'][$k] = $v;
        }
        if (jrCore_is_mobile_device() && !jrCore_is_tablet_device()) {
            $_rp['width'] = 260;
        }

        $_rp['height'] = 250;
        if (isset($params['height']) && jrCore_checktype($params['height'], 'number_nz')) {
            $_rp['height'] = (int) $params['height'];
        }
        $_rp['unique'] = uniqid();
        $out           = jrCore_parse_template('tag_cloud.tpl', $_rp, 'jrTags');
    }
    if (isset($params['assign']) && $params['assign'] != '') {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

//------------------------------------
// Functions
//------------------------------------

/**
 * Get a "clean" tag to store in the DB
 * @param string $tag
 * @return string
 */
function jrTags_get_clean_tag($tag)
{
    if (strlen("{$tag}") > 0) {
        $tag = str_replace(',', ' ', $tag);
        $tag = trim(jrCore_str_to_lower(jrCore_strip_html($tag)));
        return rawurlencode($tag);
    }
    return '';
}

/**
 * Get item_ids for a module and set of tags
 * @param string $module
 * @param array $_tags
 * @return array|bool
 */
function jrTags_get_item_ids_for_tags($module, $_tags)
{
    $mod = jrCore_db_escape($module);
    $_tg = array();
    foreach ($_tags as $t) {
        $_tg[] = jrCore_db_escape(jrTags_get_clean_tag($t));
    }
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT tag_item_id AS i FROM {$tbl} WHERE tag_module = '{$mod}' AND tag_value IN('" . implode("','", $_tg) . "')";
    $_rt = jrCore_db_query($req, 'i');
    if ($_rt && is_array($_rt)) {
        return array_keys($_rt);
    }
    return false;
}

/**
 * Check if a tag already exists for an item
 * @param string $module
 * @param int $item_id
 * @param string $tag
 * @return array|false
 */
function jrTags_tag_exists($module, $item_id, $tag)
{
    $mod = jrCore_db_escape($module);
    $tid = (int) $item_id;
    $tag = jrCore_db_escape(jrTags_get_clean_tag($tag));
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT * FROM {$tbl} WHERE tag_value = '{$tag}' AND tag_item_id = {$tid} AND tag_module = '{$mod}' LIMIT 1";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Create a new tag for an item
 * @param string $module
 * @param int $profile_id
 * @param int $item_id
 * @param string $tag
 * @return int
 */
function jrTags_create_tag($module, $profile_id, $item_id, $tag)
{
    if (empty($tag)) {
        return false;
    }
    global $_user;
    $mod = jrCore_db_escape($module);
    $uid = (int) $_user['_user_id'];
    $pid = (int) $profile_id;
    $iid = (int) $item_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "INSERT IGNORE INTO {$tbl} (tag_module, tag_profile_id, tag_item_id, tag_user_id, tag_value) VALUES ('{$mod}', {$pid}, {$iid}, {$uid}, '" . jrCore_db_escape(jrTags_get_clean_tag($tag)) . "')";
    $tid = jrCore_db_query($req, 'INSERT_ID');

    // Add Tag to item DS
    if ($pfx = jrCore_db_get_prefix($module)) {
        $_up = array();
        if ($existing = jrCore_db_get_item_key($module, $iid, "{$pfx}_tags")) {
            foreach (explode(',', $existing) as $etag) {
                $etag = trim(jrCore_str_to_lower($etag));
                if (strlen($etag) > 0) {
                    $_up[$etag] = $etag;
                }
            }
        }
        $tag       = trim(jrCore_str_to_lower($tag));
        $_up[$tag] = $tag;
        $_dt       = array(
            "{$pfx}_tags"       => ',' . trim(implode(',', $_up), ',') . ',',
            "{$pfx}_tags_count" => count($_up)
        );
        jrCore_db_update_item($module, $iid, $_dt, null, false, true, false);
    }
    jrCore_db_increment_key('jrProfile', $pid, 'profile_jrTags_tagged_item_count', 1);
    return $tid;
}

/**
 * Update an existing tag
 * @param string $module
 * @param int $item_id
 * @param string $old_tag
 * @param string $new_tag
 * @return bool
 */
function jrTags_update_tag($module, $item_id, $old_tag, $new_tag)
{
    // Update Tag Search
    $mod = jrCore_db_escape($module);
    $iid = (int) $item_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "UPDATE {$tbl} SET tag_value = '" . jrCore_db_escape(jrTags_get_clean_tag($new_tag)) . "'
             WHERE tag_module = '{$mod}' AND tag_item_id = '{$iid}' AND tag_value = '" . jrCore_db_escape(jrTags_get_clean_tag($old_tag)) . "'";
    jrCore_db_query($req);

    // Update item DS
    if ($pfx = jrCore_db_get_prefix($module)) {
        $_up = array();
        $old = trim(jrCore_str_to_lower($old_tag));
        if ($existing = jrCore_db_get_item_key($module, $iid, "{$pfx}_tags")) {
            foreach (explode(',', $existing) as $etag) {
                $etag = trim(jrCore_str_to_lower($etag));
                if (strlen($etag) > 0 && $etag != $old) {
                    $_up[$etag] = $etag;
                }
            }
        }
        $new       = trim(jrCore_str_to_lower($new_tag));
        $_up[$new] = $new;
        $_dt       = array(
            "{$pfx}_tags"       => ',' . trim(implode(',', $_up), ',') . ',',
            "{$pfx}_tags_count" => count($_up)
        );
        if (jrCore_db_update_item($module, $iid, $_dt, null, false, true, false)) {
            return true;
        }
    }
    return false;
}

/**
 * Delete an existing tag
 * @param string $module
 * @param int $profile_id
 * @param int $item_id
 * @param string $tag
 * @return bool
 */
function jrTags_delete_tag($module, $profile_id, $item_id, $tag)
{
    // Delete from tag search
    $mod = jrCore_db_escape($module);
    $iid = (int) $item_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "DELETE FROM {$tbl} WHERE tag_module = '{$mod}' AND tag_item_id = {$iid} AND tag_value = '" . jrCore_db_escape(jrTags_get_clean_tag($tag)) . "' LIMIT 1";
    jrCore_db_query($req);

    // Delete from item DS entry
    return jrTags_delete_tag_from_item_data($module, $profile_id, $item_id, $tag);
}

/**
 * Delete tag data from DS item keys
 * @param string $module
 * @param int $profile_id
 * @param int $item_id
 * @param string $tag
 * @return bool
 */
function jrTags_delete_tag_from_item_data($module, $profile_id, $item_id, $tag)
{
    if ($pfx = jrCore_db_get_prefix($module)) {
        $iid = (int) $item_id;
        if ($existing = jrCore_db_get_item_key($module, $iid, "{$pfx}_tags")) {
            $tag = rawurldecode(trim(jrCore_str_to_lower($tag)));
            if (strpos(' ' . $existing, $tag)) {
                $_up = array();
                $_ex = explode(',', $existing);
                foreach ($_ex as $etag) {
                    $etag = trim(jrCore_str_to_lower($etag));
                    if (strlen($etag) > 0 && $etag != $tag) {
                        $_up[$etag] = $etag;
                    }
                }
                $pid = (int) $profile_id;
                if (count($_up) > 0) {
                    $_dt = array(
                        "{$pfx}_tags"       => ',' . trim(implode(',', $_up), ',') . ',',
                        "{$pfx}_tags_count" => count($_up)
                    );
                    jrCore_db_update_item($module, $iid, $_dt, null, false, true, false);
                }
                else {
                    // Do this update, even though the fields are to be deleted, for the benefit of the
                    // db_update_item listener in the Search module so as to maintain the search fulltext table correctly
                    $_dt = array(
                        "{$pfx}_tags"       => '',
                        "{$pfx}_tags_count" => 0
                    );
                    jrCore_db_update_item($module, $iid, $_dt, null, false, true, false);

                    // We removed the last tag on this item
                    jrCore_db_delete_multiple_item_keys($module, $iid, array("{$pfx}_tags", "{$pfx}_tags_count"), false, true, false);
                }
                jrCore_db_decrement_key('jrProfile', $pid, 'profile_jrTags_tagged_item_count', 1);
            }
            // Fall through - tag does not exist in _tags key
        }
    }
    return true;
}

/**
 * Get all tags for an item
 * @param string $module
 * @param int $item_id
 * @return array|false
 */
function jrTags_get_tags_for_item($module, $item_id)
{
    $mod = jrCore_db_escape($module);
    $tid = (int) $item_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT * FROM {$tbl} WHERE tag_item_id = {$tid} AND tag_module = '{$mod}'";
    return jrCore_db_query($req, 'NUMERIC', false, 'tag_value');
}

/**
 * Get all tags for a profile_id
 * @param int $profile_id
 * @return array|false
 */
function jrTags_get_all_tags_for_profile_id($profile_id, $limit = 0)
{
    $_md = array();
    foreach (jrCore_get_datastore_modules() as $m => $p) {
        if (jrCore_module_is_active($m)) {
            $_md[] = jrCore_db_escape($m);
        }
    }
    $pid = (int) $profile_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT tag_id, tag_value FROM {$tbl} WHERE tag_profile_id = {$pid} AND tag_module IN('" . implode("','", $_md) . "')";
    if (is_numeric($limit) && $limit > 0) {
        $req .= " ORDER BY tag_id DESC LIMIT " . intval($limit);
    }
    return jrCore_db_query($req, 'tag_id', false, 'tag_value');
}

/**
 * Get all module tags for a profile_id
 * @param int $profile_id
 * @return array|false
 */
function jrTags_get_all_module_tags_for_profile_id($module, $profile_id, $limit = 0)
{
    $mod = jrCore_db_escape($module);
    $pid = (int) $profile_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT tag_id, tag_value FROM {$tbl} WHERE tag_profile_id = {$pid} AND tag_module = '{$mod}'";
    if (is_numeric($limit) && $limit > 0) {
        $req .= " LIMIT " . intval($limit) . " ORDER BY tag_id DESC";
    }
    return jrCore_db_query($req, 'tag_id', false, 'tag_value');
}

/**
 * Get all items tagged with a specific tag
 * @param string $tag
 * @param int $profile_id
 * @param int $pagenum
 * @param int $pagebreak
 * @return array|false
 */
function jrTags_get_profile_items_by_tag($tag, $profile_id, $pagenum, $pagebreak)
{
    $_md = array();
    foreach (jrCore_get_datastore_modules() as $m => $p) {
        if (jrCore_module_is_active($m)) {
            $_md[] = jrCore_db_escape($m);
        }
    }
    $tag = jrCore_db_escape($tag);
    $pid = (int) $profile_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT * FROM {$tbl} WHERE tag_value = '{$tag}' AND tag_profile_id = {$pid} AND tag_module IN('" . implode("','", $_md) . "') ORDER BY tag_id DESC";
    return jrCore_db_query($req, 'NUMERIC');
}

/**
 * Get all tags for a module
 * @param int $limit
 * @return array|false
 */
function jrTags_get_all_module_tags($module, $limit = 1000)
{
    $mod = jrCore_db_escape($module);
    $lim = (int) $limit;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT tag_id, tag_value FROM {$tbl} WHERE tag_module = '{$mod}' ORDER BY tag_id DESC LIMIT {$lim}";
    return jrCore_db_query($req, 'tag_id', false, 'tag_value');
}

/**
 * Get all tags
 * @param int $limit
 * @return array|false
 */
function jrTags_get_all_tags($limit = 1000)
{
    $lim = (int) $limit;
    $_md = array();
    foreach (jrCore_get_datastore_modules() as $m => $p) {
        if (jrTags_is_disabled_module($m)) {
            continue;
        }
        if (jrCore_module_is_active($m)) {
            $_md[] = jrCore_db_escape($m);
        }
    }
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT tag_id, tag_value FROM {$tbl} WHERE tag_module IN('" . implode("','", $_md) . "') ORDER BY tag_id DESC LIMIT {$lim}";
    return jrCore_db_query($req, 'tag_id', false, 'tag_value');
}

/**
 * Get tag info by tag_id
 * @param int $tag_id
 * @return array|false
 */
function jrTags_get_tag_by_tag_id($tag_id)
{
    $tid = (int) $tag_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT * FROM {$tbl} WHERE tag_id = {$tid}";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Delete a tag by tag_id
 * @param int $tag_id
 * @return int|false
 */
function jrTags_delete_tag_by_tag_id($tag_id)
{
    // Delete from tag search
    $tid = (int) $tag_id;
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "DELETE FROM {$tbl} WHERE tag_id = {$tid} LIMIT 1";
    return jrCore_db_query($req, 'COUNT');
}

/**
 * Get all tags with count
 * @param int $limit
 * @return array|false
 */
function jrTags_get_all_tags_with_count($limit = 1000)
{
    $lim = (int) $limit;
    $_md = array();
    foreach (jrCore_get_datastore_modules() as $m => $p) {
        if (jrCore_module_is_active($m)) {
            $_md[] = jrCore_db_escape($m);
        }
    }
    $tbl = jrCore_db_table_name('jrTags', 'tag_search');
    $req = "SELECT tag_value, count(tag_id) as cnt FROM {$tbl}
             WHERE tag_module IN('" . implode("','", $_md) . "')
             GROUP BY tag_value ORDER BY cnt desc LIMIT {$lim}";
    return jrCore_db_query($req, 'tag_value', false, 'cnt');
}

//------------------------------------
// Event Listeners
//------------------------------------

/**
 * Verify existing tags are inserted into new structure
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrTags_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    jrCore_queue_create('jrTags', 'verify_tags', array('verify' => 1), 0, null, 1);
    return $_data;
}

/**
 * Make sure tags get created
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrTags_db_create_item_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (!empty($_post['jr_html_form_profile_id'])) {
        if ($prefix = jrCore_db_get_prefix($_args['module'])) {
            if (!empty($_data["{$prefix}_tags"])) {
                $tbl = jrCore_db_table_name('jrTags', 'tag_search');
                $mod = jrCore_db_escape($_args['module']);
                $pid = (int) $_post['jr_html_form_profile_id'];
                $iid = (int) $_args['_item_id'];
                $uid = (int) $_user['_user_id'];
                $_rq = array();
                foreach (explode(',', $_data["{$prefix}_tags"]) as $tag) {
                    $_rq[] = "INSERT IGNORE INTO {$tbl} (tag_module, tag_profile_id, tag_item_id, tag_user_id, tag_value)
                              VALUES ('{$mod}', {$pid}, {$iid}, {$uid}, '" . jrCore_db_escape(jrTags_get_clean_tag($tag)) . "')";
                }
                if (count($_rq) > 0) {
                    jrCore_db_multi_query($_rq, false);
                }
            }
        }
    }
    return $_data;
}

/**
 * Make sure tags get updated
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrTags_db_update_item_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (!empty($_post['jr_html_form_profile_id'])) {
        if ($prefix = jrCore_db_get_prefix($_args['module'])) {
            if (!empty($_data["{$prefix}_tags"])) {
                $tbl = jrCore_db_table_name('jrTags', 'tag_search');
                $mod = jrCore_db_escape($_args['module']);
                $pid = (int) $_post['jr_html_form_profile_id'];
                $iid = (int) $_args['_item_id'];
                $uid = (jrUser_is_logged_in() && !empty($_user['_user_id'])) ? intval($_user['_user_id']) : 0;
                $_rq = array("DELETE FROM {$tbl} WHERE tag_item_id = {$iid} AND tag_profile_id = {$pid} AND tag_module = '{$mod}'");
                foreach (explode(',', $_data["{$prefix}_tags"]) as $tag) {
                    $_rq[] = "INSERT IGNORE INTO {$tbl} (tag_module, tag_profile_id, tag_item_id, tag_user_id, tag_value)
                              VALUES ('{$mod}', {$pid}, {$iid}, {$uid}, '" . jrCore_db_escape(jrTags_get_clean_tag($tag)) . "')";
                }
                if (count($_rq) > 0) {
                    jrCore_db_multi_query($_rq, false);
                }
            }
        }
    }
    return $_data;
}

/**
 * Clean up tag searches
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrTags_db_search_params_listener($_data, $_user, $_conf, $_args, $event)
{
    // @deprecated but still supported:
    // tags are stored like this: audio_tags = ",awesome,fast,good drums,"
    // search1="audio_tags = awesome"
    // search1="audio_tags != awesome"
    // gets translated to this:
    // search1="audio_tags like ,awesome,"
    // search1="audio_tags not_like ,awesome,"

    // New method:
    // audio_tags = awesome song - exact match single
    // audio_tags in awesome song,fast beat and guitar,vocal only - exact match multiple
    // audio_tags != awesome - exclude exact match single
    // audio_tags not_in awesome song,fast beat and guitar,vocal only - exclude exact match multiple

    $pfx = jrCore_db_get_prefix($_args['module']);
    // search for any searches on "{$module_prefix}_tags"
    if ($pfx && isset($_data['search']) && is_array($_data['search'])) {
        foreach ($_data['search'] as $k => $ss) {
            // Make sure it is a match for our module and not already setup
            if (strpos($ss, "{$pfx}_tags ") === 0 && !strpos(' ' . $ss, '%')) {
                // We have a tags search - fix it up
                $ss = preg_replace('!\s+!', ' ', $ss);
                list(, $op, $st) = explode(' ', $ss, 3);
                $st = trim(trim($st), ',');
                switch (trim($op)) {

                    case '=':
                        if ($_ids = jrTags_get_item_ids_for_tags($_args['module'], array($st))) {
                            $_data['search'][$k] = '_item_id in ' . implode(',', $_ids);
                            unset($_ids);
                        }
                        break;

                    case '!=':
                        if ($_ids = jrTags_get_item_ids_for_tags($_args['module'], array($st))) {
                            $_data['search'][$k] = '_item_id not_in ' . implode(',', $_ids);
                            unset($_ids);
                        }
                        break;

                    // We've been given multiple tags
                    case 'in':
                        if ($_tags = explode(',', $st)) {
                            if ($_ids = jrTags_get_item_ids_for_tags($_args['module'], $_tags)) {
                                $_data['search'][$k] = '_item_id in ' . implode(',', $_ids);
                                unset($_ids);
                            }
                        }
                        break;

                    case 'not_in':
                        if ($_tags = explode(',', $st)) {
                            if ($_ids = jrTags_get_item_ids_for_tags($_args['module'], $_tags)) {
                                $_data['search'][$k] = '_item_id not_in ' . implode(',', $_ids);
                                unset($_ids);
                            }
                        }
                        break;
                }
            }
        }
    }
    return $_data;
}

/**
 * Get DataStore modules that support Tags
 * @return array|false
 */
function jrTags_get_supported_modules()
{
    global $_mods;
    $_ot = array();
    foreach ($_mods as $mod => $_inf) {
        if (jrCore_is_datastore_module($mod)) {
            $_ot[$mod] = $_inf['module_name'];
        }
    }
    if (count($_ot) > 0) {
        natcasesort($_ot);
        return $_ot;
    }
    return false;
}

/**
 * Get any modules that have been disabled
 * @return array|false
 */
function jrTags_get_disabled_modules()
{
    if ($_disabled = jrCore_get_config_value('jrTags', 'disabled', false)) {
        $_disabled = explode(',', $_disabled);
        return array_combine($_disabled, $_disabled);
    }
    return false;
}

/**
 * Return TRUE if a module is a disabled module
 * @param string $module Module
 * @return false|int
 */
function jrTags_is_disabled_module($module)
{
    if ($disabled = jrCore_get_config_value('jrTags', 'disabled', false)) {
        $disabled = trim(trim($disabled, ','));
        return strpos(' ' . ",{$disabled},", ",{$module},");
    }
    return false;
}

//-----------------------------------
// Form Field
//-----------------------------------

/**
 * Display tags in a form
 * @param array $_field Array of Field parameters
 * @param array $_att Additional HTML parameters
 * @return bool
 *
 * // Sensei
 * $_opt = array(
 *     'michael ussher',
 *     'christian tissier',
 *     'iire sensei',
 * );
 * $_tmp = array(
 *     'name'        => 'seminar_teacher',
 *     'label'       => 'Sensei',
 *     'placeholder' => 'Sensei',
 *     'help'        => 'The Sensei instructing the class',
 *     'type'        => 'tags',
 *     'options'     => $_opt,
 *     'required'    => true
 * );
 * jrCore_form_field_create($_tmp);
 */
function jrTags_form_field_tags_display($_field, $_att = null)
{
    global $_mods;
    if (!jrCore_get_flag('jrtags_tagit_included')) {
        $_tm = array('source' => jrCore_get_base_url() . "/modules/jrTags/contrib/tagit/jquery.tagit.js?v={$_mods['jrTags']['module_updated']}");
        jrCore_create_page_element('javascript_href', $_tm);
        $_tm = array('source' => jrCore_get_base_url() . "/modules/jrTags/contrib/tagit/jquery.tagit.css?v={$_mods['jrTags']['module_updated']}");
        jrCore_create_page_element('css_href', $_tm);
        jrCore_set_flag('jrtags_tagit_included', 1);
    }
    $cls = 'form_text form_tags ' . jrCore_form_field_get_hilight($_field['name']);
    if (!empty($_field['class'])) {
        $cls .= ' ' . $_field['class'];
    }
    // Get our tab index
    if (!empty($_field['placeholder'])) {
        $plc = addslashes($_field['placeholder']);
    }
    else {
        $_ln = jrUser_load_lang_strings();
        $plc = $_ln['jrTags'][24];
    }
    $idx = jrCore_form_field_get_tab_index($_field);
    $htm = '<ul id="' . $_field['name'] . '_select" class=" ' . $cls . '" tabindex="' . $idx . '" placeholder="' . $plc . '"';
    if (is_array($_att)) {
        foreach ($_att as $key => $attr) {
            $htm .= ' ' . $key . '="' . $attr . '"';
        }
    }
    $htm .= '>';
    if (!empty($_field['value'])) {
        if ($_tags = explode(',', $_field['value'])) {
            if (!empty($_tags)) {
                foreach ($_tags as $tag) {
                    if (strlen($tag) === 0) {
                        continue;
                    }
                    $htm .= "<li>{$tag}</li>";
                }
            }
        }
    }
    $_field['html']     = $htm;
    $_field['type']     = 'text';
    $_field['template'] = 'form_field_elements.tpl';
    jrCore_create_page_element('page', $_field);

    // initialize tags
    if (is_array($_field['options'])) {
        $_js = array("var tags = " . json_encode($_field['options']) . "; $(\"#{$_field['name']}_select\").tagit({ tags: tags, field: \"{$_field['name']}\" });");
    }
    else {
        $_js = array("$('#{$_field['name']}_select').tagit()");
    }
    jrCore_create_page_element('javascript_ready_function', $_js);
    return true;
}

/**
 * Defines Form Designer field options
 * @return array
 */
function jrTags_form_field_tags_form_designer_options()
{
    return array(
        'disable_options' => true
    );
}

/**
 * jrCore_form_field_select_and_text_validate
 * @param array $_field Array of form field info
 * @param array $_post Global $_post from jrCore_parse_url()
 * @param string $e_msg Error message to use in validation
 * @return array
 */
function jrTags_form_field_tags_assembly($_field, $_post)
{
    $name = $_field['name'];
    if (!empty($_post['tag']) && is_array($_post['tag'])) {
        $tmp          = array_map('trim', $_post['tag']);
        $_post[$name] = implode(',', $tmp);
    }
    return $_post;
}

/**
 * Additional form field HTML attributes that can be passed in via the form
 * @return array
 */
function jrTags_form_field_tags_attributes()
{
    return array('disabled', 'readonly', 'maxlength', 'onfocus', 'onblur', 'onselect', 'onkeypress', 'onkeyup', 'style', 'class', 'autocorrect', 'autocapitalize', 'placeholder');
}
