<?php
 /**
 * Jamroom Media URL Scanner module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

// our default user agent
const JRURLSCAN_USER_AGENT = 'Mozilla/5.0 (Linux; Android 6.0.1; Nexus 5X Build/MMB29P) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.5735.179 Mobile Safari/537.36 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)';
const JRURLSCAN_TEXT_KEY   = 'jrurlscan_replace_player_text';
const JRURLSCAN_URLS_KEY   = 'jrurlscan_replace_player_urls';

/**
 * meta
 */
function jrUrlScan_meta()
{
    return array(
        'name'        => 'Media URL Scanner',
        'url'         => 'urlscan',
        'version'     => '2.2.1',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Scans text and converts supported media URLs into an inline viewer',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/2894/media-url-scanner',
        'priority'    => 20,
        'license'     => 'jcl',
        'requires'    => 'jrCore:6.5.12,jrSystemTools',
        'recommended' => 'jrShareThis',
        'category'    => 'site'
    );
}

/**
 * init
 */
function jrUrlScan_init()
{
    // Register the module's javascript
    jrCore_register_module_feature('jrCore', 'javascript', 'jrUrlScan', 'jrUrlScan.js');
    jrCore_register_module_feature('jrCore', 'css', 'jrUrlScan', 'jrUrlScan.css');

    // register the module's trigger
    jrCore_register_event_trigger('jrUrlScan', 'all_found_urls', 'Fired with an array of all URLs found in the text');
    jrCore_register_event_trigger('jrUrlScan', 'url_found', 'Fired when a URL is found in any text');
    jrCore_register_event_trigger('jrUrlScan', 'url_player_params', 'Fired with template params when parsing urlscan_player.tpl');
    jrCore_register_event_trigger('jrUrlScan', 'get_url_card', 'Fired with a URL to get URL card');
    jrCore_register_event_trigger('jrUrlScan', 'get_tags_for_url', 'Fired with a URL to get OG Tags');
    jrCore_register_event_trigger('jrUrlScan', 'found_url_tags', 'Fired with tags for a URL');

    // Watch for URLs in new timeline entries
    jrCore_register_event_listener('jrCore', 'view_results', 'jrUrlScan_view_results_listener');
    jrCore_register_event_listener('jrCore', 'db_create_item_data', 'jrUrlScan_db_create_item_data_listener');
    jrCore_register_event_listener('jrCore', 'db_search_params', 'jrUrlScan_db_search_params_listener');
    jrCore_register_event_listener('jrCore', 'hourly_maintenance', 'jrUrlScan_hourly_maintenance_listener');
    jrCore_register_event_listener('jrCore', 'format_string_display', 'jrUrlScan_format_string_display_listener');

    // We provide a string formatter
    $_tmp = array(
        'wl'    => 'urlscan',
        'label' => 'Media URL Scanner',
        'help'  => 'If active, text is scanned for media urls and if found they are replaced with an embedded player'
    );
    jrCore_register_module_feature('jrCore', 'format_string', 'jrUrlScan', 'jrUrlScan_format_string', $_tmp);

    // When we are enabled, we must disable the Core provided URL converter
    $_tmp = jrCore_get_flag('jrcore_register_module_feature');
    unset($_tmp['jrCore']['format_string']['jrCore']['jrCore_format_string_clickable_urls']);
    jrCore_set_flag('jrcore_register_module_feature', $_tmp);

    // Our URL scan workers
    jrCore_register_queue_worker('jrUrlScan', 'url_update', 'jrUrlScan_url_update_worker', 0, 1, 120, LOW_PRIORITY_QUEUE);

    jrCore_register_module_feature('jrCore', 'tool_view', 'jrUrlScan', 'browser', array('URL Card Browser', 'View and Refresh existing URL Cards'));

    return true;
}

//-----------------------
// QUEUE WORKERS
//-----------------------

/**
 * Update URL Info
 * @param $_queue
 * @return bool
 */
function jrUrlScan_url_update_worker($_queue)
{
    if (is_array($_queue)) {
        jrUrlScan_update_url_card($_queue);
    }
    return true;
}

//-----------------------
// EVENT LISTENERS
//-----------------------

/**
 * Strip tracking params from shared URLs
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrUrlScan_db_create_item_data_listener($_data, $_user, $_conf, $_args, $event)
{
    if (jrCore_get_config_value('jrUrlScan', 'strip_tracking', 'on') == 'on') {
        if ($_args['module'] == 'jrAction' && !empty($_data['action_text'])) {
            preg_match_all('~[^">=]https?://[^\s()<>]+(?:\([\w]+\)|([^[:punct:]\s]|[/=#&-]))~', ' ' . $_data['action_text'], $_found);
            $_found = $_found[0];
            if ($_found && is_array($_found) && count($_found) > 0) {
                $_found = array_unique($_found);
                foreach ($_found as $url) {
                    // valid URL will have at least 1 period in it
                    if (!strpos($url, '.') || strpos($url, 'urlscan_player')) {
                        continue;
                    }
                    $_data['action_text'] = str_replace(trim($url), jrUrlScan_get_trimmed_url($url), $_data['action_text']);
                }
            }
        }
    }
    return $_data;
}

/**
 * Verify URLs
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrUrlScan_hourly_maintenance_listener($_data, $_user, $_conf, $_args, $event)
{
    $num = (int) jrCore_get_config_value('jrUrlScan', 'daily_maintenance', 480);
    if ($num > 0) {
        $num = (int) ceil($num / 24);
        $_sp = array(
            'search'          => array(
                '_updated < ' . (time() - 86400)
            ),
            'skip_all_checks' => true,
            'limit'           => $num
        );
        $_sp = jrCore_db_search_items('jrUrlScan', $_sp);
        if ($_sp && is_array($_sp) && isset($_sp['_items'])) {
            foreach ($_sp['_items'] as $_url) {
                jrCore_queue_create('jrUrlScan', 'url_update', $_url);
            }
        }
    }
    return $_data;
}

/**
 * Add URL card Javascript in on timeline
 * @param $_data string incoming HTML
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return string
 */
function jrUrlScan_view_results_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (jrUser_is_logged_in() && jrCore_module_is_active('jrAction') && strpos($_data, 'action_update')) {
        if (!isset($_post['option']) || $_post['option'] == jrCore_get_module_url('jrAction')) {
            $html  = '<script type="text/javascript">$(document).ready(function(){jrUrlScan_init_url_listener(\'#action_update\')});</script><div id="urlscan_target"></div>';
            $_data = str_replace('</body>', "\n{$html}</body>", $_data);
        }
    }
    return $_data;
}

/**
 * Add URL card Javascript in on timeline
 * @param $_data mixed information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrUrlScan_format_string_display_listener($_data, $_user, $_conf, $_args, $event)
{
    // Replace any found URL Scan players
    if (!empty($_data['string'])) {
        if ($_rp = jrCore_get_flag(JRURLSCAN_TEXT_KEY)) {
            $_data['string'] = str_replace(array_keys($_rp), $_rp, $_data['string']);
            jrCore_delete_flag(JRURLSCAN_TEXT_KEY);
        }
        if ($_rp = jrCore_get_flag(JRURLSCAN_URLS_KEY)) {
            $_data['string'] .= "\n" . '<script type="text/javascript">$(document).ready(function(){' . implode('', $_rp) . '});</script>';
            jrCore_delete_flag(JRURLSCAN_URLS_KEY);
        }
    }
    return $_data;
}

/**
 * Watch for jrAction jrCore_list calls
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrUrlScan_db_search_params_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_args['module']) && $_args['module'] == 'jrAction') {
        jrCore_set_flag('jrurlscan_expand_url_cards', 1);
    }
    return $_data;
}

//-----------------------
// FUNCTIONS
//-----------------------

/**
 * Return TRUE of OpenGraph.io is enabled and configured
 * @return bool
 */
function jrUrlScan_opengraph_is_enabled()
{
    if (jrCore_get_config_value('jrUrlScan', 'opengraph_enabled', 'disabled') != 'disabled') {
        if (jrCore_get_config_value('jrUrlScan', 'opengraph_api_key', false)) {
            return true;
        }
    }
    return false;
}

/**
 * Searches for URLs in a text string.
 * If found, links them to a 'player div' under the text
 * @param string $string string to search
 * @param int $quota_id Quota ID of item owner
 * @return string
 */
function jrUrlScan_format_string($string, $quota_id = 0)
{
    return jrUrlScan_replace_urls($string);
}

/**
 * Scan Text and replace URLs with embedded players
 * @param $text
 * @return string
 */
function jrUrlScan_replace_urls($text)
{
    global $_conf, $_post, $_urls;

    // Extract any urls
    $_found = array();
    if (strpos(' ' . $text, 'http')) {

        if (strpos($text, 'http') === 0) {
            $text = ' ' . $text;
        }

        // Fix for URLs right after an opening HTML tag which can be added by the editor
        $text = str_replace('>', '> ', $text);

        if (strpos(' ' . $text, '(http')) {
            $text = str_replace('(http', '( http', $text);
        }

        preg_match_all('~[^">=]https?://[^\s()<>]+(?:\([\w]+\)|([^[:punct:]\s]|[/=#&-]))~', $text, $_found);
        $_found = $_found[0];
        $fcount = count($_found);
        if (is_array($_found) && $fcount > 0) {

            // Sort shortest to longest
            if ($fcount > 1) {
                usort($_found, function ($a, $b) {
                    return (strlen($a) < strlen($b)) ? -1 : 1;
                });
            }

            // Process each URL
            $_tmp = array(
                '_items' => array()
            );
            // trim any leading characters off of the URL. eg: ?http://
            foreach ($_found as $i => $url) {
                $http = strpos($url, 'http');
                if ($http !== 0) {
                    $_found[$i] = substr($url, $http);
                }
            }

            $_found = array_unique($_found);
            foreach ($_found as $i => $url) {

                // valid URL will have at least 1 period in it
                if (!strpos($url, '.')) {
                    unset($_found[$i]);
                    continue;
                }

                $url = jrUrlScan_get_trimmed_url($url);
                if (strlen($url) === 0) {
                    unset($_found[$i]);
                    continue;
                }
                // Have we already processed this URL?
                if (strpos($url, 'urlscan_player')) {
                    unset($_found[$i]);
                    continue;
                }
                $_args = array(
                    'url' => $url,
                    'i'   => $i
                );
                // Trigger event so the listening module can provide us with the load URL
                $_tmp = jrCore_trigger_event('jrUrlScan', 'url_found', $_tmp, $_args);
                if (!isset($_tmp['_items'][$i]) || !isset($_tmp['_items'][$i]['url'])) {
                    $create = false;
                    if (isset($_post['module']) && $_post['module'] == 'jrAction') {
                        // We're viewing an action timeline - create cards
                        $create = true;
                    }
                    elseif (isset($_post['option']) && isset($_urls["{$_post['option']}"]) && $_urls["{$_post['option']}"] == 'jrAction') {
                        $create = true;
                    }
                    elseif (jrCore_get_flag('jrurlscan_expand_url_cards')) {
                        $create = true;
                    }
                    if ($create) {
                        // fallback to og:tags and page info scrape.
                        $_tmp = jrUrlScan_get_card_info_for_url($_tmp, $_args);
                        if ($_tmp && is_array($_tmp) && isset($_tmp['_item'])) {
                            $_found[$i]['_item'] = $_tmp['_item'];
                        }
                    }
                }
            }

            // Trigger for what we have come up with
            $_tmp = jrCore_trigger_event('jrUrlScan', 'all_found_urls', $_tmp, $_found);

            $_ui = array();
            $_tx = array();
            foreach ($_found as $i => $url) {

                $key = jrCore_create_unique_string(8);
                $url = jrUrlScan_get_trimmed_url($url);
                $did = "url-scan-{$key}";

                // Did we get a converted URL?
                if (is_array($_tmp['_items'][$i]) && isset($_tmp['_items'][$i]['load_url'])) {

                    $pattern = '`[^">]' . preg_quote($url) . '`';

                    // Is this a URL card?  We always expand URL cards
                    if (strpos($_tmp['_items'][$i]['load_url'], '/ogtags/')) {
                        $card      = jrCore_parse_template('url_card.tpl', array('item' => $_tmp['_items'][$i]['_item']), 'jrUrlScan');
                        $text      = preg_replace($pattern, $key, $text);
                        $_tx[$key] = "<div id=\"{$did}\" class=\"urlscan_block urlscan_card_block\">" . str_replace("\n", "", $card) . "</div>";
                    }

                    // Player - immediate load - NOT IN TICKETS
                    elseif ((!isset($_post['module']) || $_post['module'] !== 'jrTicket') && isset($_conf['jrUrlScan_immediate_replace']) && $_conf['jrUrlScan_immediate_replace'] == 'on') {
                        $text      = preg_replace($pattern, $key, $text);
                        $_ui[$key] = "$('#{$did}').load('{$_tmp['_items'][$i]['load_url']}', function() { $('#{$did}').slideDown(200); });";
                        $_tx[$key] = "<div id=\"{$did}\" class=\"urlscan_block\"></div>";
                    }

                    // Players - load on click
                    else {
                        $event = 'onclick';
                        if (jrCore_is_mobile_device()) {
                            $event = 'ontouchend';
                        }
                        if (jrCore_checktype($_conf['jrUrlScan_play_button_size'], 'number_nz')) {
                            $text = preg_replace($pattern, "<a {$event}=\"jrUrlScan_load_player('{$_tmp['_items'][$i]['load_url']}',this)\"><img src=\"{$_conf['jrCore_base_url']}/image/img/module/jrUrlScan/button_play.png\" width=\"{$_conf['jrUrlScan_play_button_size']}\" height=\"{$_conf['jrUrlScan_play_button_size']}\" alt=\"play\"> {$_tmp['_items'][$i]['title']}</a>", $text);
                        }
                        else {
                            $text      = preg_replace($pattern, $key, $text);
                            $title     = (!empty($_tmp['_items'][$i]['title'])) ? $_tmp['_items'][$i]['title'] : 'Click to View Video';
                            $_tx[$key] = "<a class=\"urlscan_card_load\" {$event}=\"jrUrlScan_load_player('{$_tmp['_items'][$i]['load_url']}',this)\">{$title}</a>";
                        }
                    }
                }

                // Unsafe URLs
                elseif (is_array($_tmp['_items'][$i]) && isset($_tmp['_items'][$i]['unsafe'])) {

                    // We have an unsafe URL
                    $char = '%';
                    if (strpos($url, $char)) {
                        $char = '~';
                    }
                    if (jrUser_is_admin()) {
                        $text = preg_replace("{$char}(" . preg_quote($url) . ")([ \t\n\r\)\.,<]){$char}", "<b>Reported Malware URL</b>: $1", $text . ' ');
                    }
                    else {
                        if (!isset($_ln)) {
                            $_ln = jrUser_load_lang_strings();
                        }
                        $text = preg_replace("{$char}(" . preg_quote($url) . ")([ \t\n\r\)\.,<]){$char}", "<b>{$_ln['jrUrlScan'][2]}</b>", $text . ' ');
                    }

                }

                // Not Found URLs
                elseif (is_array($_tmp['_items'][$i]) && isset($_tmp['_items'][$i]['not_found'])) {

                    // We have an unsafe URL
                    $char = '%';
                    if (strpos($url, $char)) {
                        $char = '~';
                    }
                    if (jrUser_is_admin()) {
                        $text = preg_replace("{$char}(" . preg_quote($url) . ")([ \t\n\r\)\.,<]){$char}", "<b>Dead URL</b>: $1", $text . ' ');
                    }
                    else {
                        // don't hyperlink to a Dead URL
                        if (!isset($_ln)) {
                            $_ln = jrUser_load_lang_strings();
                        }
                        $text = preg_replace("{$char}(" . preg_quote($url) . ")([ \t\n\r\)\.,<]){$char}", "{$_ln['jrUrlScan'][3]} $1", $text . ' ');
                    }

                }
                else {

                    // Standard - create clickable URLs
                    $char = '%';
                    if (strpos($url, $char)) {
                        $char = '~';
                    }
                    $temp = preg_replace("{$char}(" . preg_quote($url) . ")([ \t\n\r\)\.<]|,[\n\r ]){$char}", "<a href=\"$1\" target=\"_blank\" rel=\"nofollow\">$1</a>$2", $text . ' ');
                    if (strlen($temp) > strlen($url)) {
                        $text = $temp;
                    }
                }
            }

            // Note: We save all player replacements into a global key - these
            // will be replaced in the HTML in the view_results listener
            // This ensure other string formatters don't mess with our players
            if (count($_ui) > 0) {
                if (!$_rp = jrCore_get_flag(JRURLSCAN_URLS_KEY)) {
                    $_rp = array();
                }
                $_rp = $_rp + $_ui;
                jrCore_set_flag(JRURLSCAN_URLS_KEY, $_rp);
            }
            if (count($_tx) > 0) {
                if (!$_rp = jrCore_get_flag(JRURLSCAN_TEXT_KEY)) {
                    $_rp = array();
                }
                $_rp = $_rp + $_tx;
                jrCore_set_flag(JRURLSCAN_TEXT_KEY, $_rp);
            }
        }
    }
    return $text;
}

/**
 * Get a trimmed URL for storage in database
 * @param string $url
 * @return string
 */
function jrUrlScan_get_trimmed_url($url)
{
    $url = trim(trim($url, "'"));
    if (jrCore_get_config_value('jrUrlScan', 'strip_tracking', 'on') == 'on') {
        $url = jrUrlScan_strip_url_tracking_params($url);
    }
    return $url;
}

/**
 * Extract Meta Tags from a URL
 * @param $url string URL
 * @param $agent string wget User Agent
 * @param $skip_check bool allow the retrieval of og:tags for modules that provide their own.
 * @return array|bool
 */
function jrUrlScan_extract_tags_from_url($url, $agent = null, $skip_check = false)
{
    global $_conf, $_urls;
    if (!jrCore_checktype($url, 'url')) {
        return false;
    }

    if (!$skip_check) {
        // Some local modules provide a "player" for URL Scan
        // If this is a player URL, we return FALSE so the module can handle it
        if (strpos($url, $_conf['jrCore_base_url']) === 0) {
            $_tm = parse_url($url);
            if ($_tm && is_array($_tm) && isset($_tm['path']) && strlen($_tm['path']) > 1) {
                $mod         = false;
                $_tm['path'] = trim($_tm['path'], '/');
                $_tm         = explode('/', $_tm['path']);
                if (isset($_tm[0]) && isset($_urls["{$_tm[0]}"])) {
                    $mod = $_urls["{$_tm[0]}"];
                }
                elseif (isset($_tm[1]) && isset($_urls["{$_tm[1]}"])) {
                    $mod = $_urls["{$_tm[1]}"];
                }
                if ($mod && is_file(APP_DIR . "/modules/{$mod}/templates/urlscan_player.tpl")) {
                    // This module handles it's own URLs
                    return false;
                }
            }
            else {
                return false;
            }
        }
    }

    // Allow modules to extract their own tags (i.e. by API, etc.)
    $tags  = array();
    $_data = array(
        'url'   => $url,
        'agent' => $agent,
        'tags'  => $tags
    );
    $_data = jrCore_trigger_event('jrUrlScan', 'get_tags_for_url', $_data);
    if (isset($_data['tags']) && is_array($_data['tags']) && count($_data['tags']) > 0) {
        // Our listener created our OG tags for us
        return $_data['tags'];
    }
    unset($_data);

    // Are we are grabbing tags via OpenGraph.io?
    if (jrCore_get_config_value('jrUrlScan', 'opengraph_enabled', 'disabled') == 'primary' && jrUrlScan_opengraph_is_enabled()) {

        // Yes - OpenGraph.io is PRIMARY
        $tags = jrUrlScan_get_url_tags_with_opengraph($url);

    }
    else {

        // Using Wget (local)
        jrCore_record_event('jrUrlScan', 'get_tags');
        $html = jrUrlScan_get_url_with_wget($url, $error);
        if (!empty($error) && strpos($url, 'https') === 0 && stripos($error, 'unable to establish SSL')) {
            // try again without SSL
            $url  = str_replace('https:', 'http:', $url);
            $html = jrUrlScan_get_url_with_wget($url, $error);
            if (!empty($error) && stripos($error, 'unable to establish SSL')) {
                $html = false;
            }
        }

        if (strlen($error) > 0) {

            // Permanent errors:
            // failed: Name or service not known
            // unable to resolve host address
            // ERROR 404: Not Found
            $_checks = array(
                'name or service not known',
                'unable to resolve host',
                'not found'
            );
            foreach ($_checks as $check) {
                if (stripos($error, $check)) {
                    return 404;
                }
            }

            // Possible temp errors:
            // Connection timed out
            // ERROR 403: Forbidden
            $_checks = array(
                'timed out',
                '403: forbidden'
            );
            foreach ($_checks as $check) {
                if (stripos($error, $check)) {
                    return 'retry';
                }
            }

        }

        // Strip out unnecessary JS
        // This is done since extensive amounts of embedded JS can mess up DOMDocument
        if (!empty($html)) {
            $html = preg_replace('~<script\b[^>]*>(.*?)</script>~is', '', $html);
        }

        // Results are too small to have OG Tags
        if (empty($html) || strlen($html) < 100) {
            if (jrCore_get_config_value('jrUrlScan', 'opengraph_enabled', 'disabled') == 'fallback' && jrUrlScan_opengraph_is_enabled()) {
                if ($tags = jrUrlScan_get_url_tags_with_opengraph($url)) {
                    return $tags;
                }
                return $tags;
            }
        }

        // Use our document parser to get meta tags and images
        if (class_exists('DOMDocument')) {

            $doc = new DOMDocument();
            if (preg_match_all('/<meta([^>]+)content=["\']([^>]+)>/', $html, $matches)) {

                // @note: we use @here since HTML can be malformed
                @$doc->loadHTML('<?xml encoding="utf-8" ?>' . implode($matches[0]));
                foreach ($doc->getElementsByTagName('meta') as $meta) {
                    $name = $meta->getAttribute('name');
                    if (!empty($name)) {
                        if (!isset($tags[$name])) {
                            $tags[$name] = $meta->getAttribute('content');
                        }
                        else {
                            $tags[$name] .= ' ' . $meta->getAttribute('content');
                        }
                    }
                    else {
                        $name = $meta->getAttribute('property');
                        if (!empty($name)) {
                            if (!isset($tags[$name])) {
                                $tags[$name] = $meta->getAttribute('content');
                            }
                            else {
                                $tags[$name] .= ' ' . $meta->getAttribute('content');
                            }
                        }
                    }
                }
            }

            // Did we get an image?
            // Possible image tags:
            // [og:image]
            // [twitter:image]
            // [msapplication-TileImage]
            $get_image = true;
            if (!empty($tags)) {
                foreach (array('og:image', 'twitter:image', 'msapplication-TileImage') as $i) {
                    if (isset($tags[$i])) {
                        $get_image = false;
                        break;
                    }
                }
            }
            if ($get_image) {
                // If there is no image, see if we can find the site logo
                if (preg_match_all('/img[^>]*src=["\']([^"\']+)["\']/', $html, $matches)) {
                    foreach ($matches[1] as $src) {
                        if (stripos(' ' . $src, 'logo') && strpos($src, 'http') === 0) {
                            $tags['og:image'] = $src;
                            break;
                        }
                    }
                }
            }

            // make sure we're only saving one URL for the og:image
            if (isset($tags['og:image']) && strpos($tags['og:image'], ' ')) {
                // if more than one og:image was found on the page it will be concatenated, use the first valid URL
                $_iurl = explode(' ', $tags['og:image']);
                foreach ($_iurl as $u) {
                    if (jrCore_checktype($u, 'url')) {
                        $tags['og:image'] = $u;
                        break;
                    }
                }
            }
        }

        // If there is no OG:title see if we can grab the page title
        if (!isset($tags['og:title'])) {
            // Look for page title
            if (preg_match("/<title[^>]*>(.*)<\/title>/siU", $html, $match)) {
                $tags['og:title'] = trim(preg_replace('/\s+/', ' ', $match[1]));
            }
        }

        // if og:image is set and begins with // then prefix with http://
        if (!empty($tags['og:image']) && strpos($tags['og:image'], '//') === 0) {
            $prefix           = strpos($url, 'https') ? 'https:' : 'http:';
            $tags['og:image'] = $prefix . $tags['og:image'];
        }
    }

    // Return found tags via trigger
    return jrCore_trigger_event('jrUrlScan', 'found_url_tags', $tags, array('url' => $url));
}

/**
 * Get URL tags using OpenGraph.io
 * @param string $url URL
 * @return array|false
 */
function jrUrlScan_get_url_tags_with_opengraph($url)
{
    // https://opengraph.io/api/1.1/site/<URL encoded site URL>?app_id=<API Key>
    $url = urlencode($url);
    $api = jrCore_get_config_value('jrUrlScan', 'opengraph_api_key', false);
    $url = "https://opengraph.io/api/1.1/site/{$url}?app_id={$api}";
    if ($_rs = jrCore_load_url($url)) {
        if ($_rs = json_decode($_rs, true)) {

            // Get the data we need
            if (!empty($_rs['openGraph'])) {
                $tags = array();
                foreach ($_rs['openGraph'] as $k => $v) {
                    if (!is_array($v)) {
                        $tags["og:{$k}"] = $v;
                    }
                    else {
                        foreach ($v as $kk => $vv) {
                            if ($kk == 'url') {
                                $tags["og:{$k}"] = $vv;
                            }
                            else {
                                $tags["og:{$k}:{$kk}"] = $vv;
                            }
                        }
                    }
                }
                return $tags;
            }
        }
    }
    return false;
}

/**
 * Get wget binary
 * @return bool|string
 */
function jrUrlScan_get_wget_binary()
{
    // @note we use the system one first since it likely supports newer SSL ciphers (non tls1)
    $wget = jrCore_get_tool_path('wget', 'jrUrlScan');
    if ((!$wget || strpos($wget, '/tools/')) && is_file('/usr/bin/wget') && is_executable('/usr/bin/wget')) {
        // We are setup to use the default jrSystemTools binary - use system provided
        $wget = '/usr/bin/wget';
    }
    return $wget;
}

/**
 * Get the contents of a URL using wget
 * @param string $url URL to load
 * @param string $errors array of errors encountered in URL load
 * @param string $cookies array of cookies set in URL load
 * @param int $timeout Max time wget can run, in seconds
 * @return bool|string
 */
function jrUrlScan_get_url_with_wget($url, &$errors = '', &$cookies = '', $timeout = 5)
{
    if ($wget = jrUrlScan_get_wget_binary()) {

        $file  = jrCore_get_module_cache_dir('jrUrlScan');
        $file  = "{$file}/" . jrCore_create_unique_string(12);
        $errs  = "{$file}.errors";
        $temp  = "{$file}.cookies";
        $agent = jrUrlScan_get_user_agent();

        // [HTTP_CONNECTION] => keep-alive
        // [HTTP_CACHE_CONTROL] => max-age=0
        // [HTTP_USER_AGENT] => Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_3) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/80.0.3987.106 Safari/537.36
        // [HTTP_ACCEPT] => text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9
        // [HTTP_ACCEPT_ENCODING] => gzip, deflate, br
        // [HTTP_ACCEPT_LANGUAGE] => en-US,en;q=0.9
        $enc = '';
        if (function_exists('gzdecode')) {
            $enc = " --header=\"Accept-Encoding: gzip, deflate\"";
        }

        $out = null;
        $err = null;
        ob_start();
        jrCore_run_module_function('jrCloudClient_start_timer', 'jrUrlScan', 'get_tags');
        $cmd = "timeout " . intval($timeout) . " {$wget} -O {$file} --user-agent=" . escapeshellarg($agent) . " --header=\"Connection: keep-alive\" --header=\"Cache-Control: max-age=0\" --header=\"Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9\" --header=\"Accept-Language: en-US,en;q=0.5\"{$enc} --save-cookies={$temp} --keep-session-cookies --max-redirect=3 --timeout=" . intval($timeout) . " --tries=1 --no-verbose --no-check-certificate " . escapeshellarg($url) . " 2>{$errs}";
        jrCore_run_timed_command($cmd, ($timeout + 1), $out, $err);
        jrCore_run_module_function('jrCloudClient_stop_timer', 'jrUrlScan', 'get_tags');
        ob_end_clean();

        if (!is_file($file)) {
            return false;
        }
        $size = filesize($file);
        if ($size < 100) {
            jrCore_unlink($file);
            jrCore_unlink($errs);
            jrCore_unlink($temp);
            return false;
        }

        // We have to actually parse for errors since wget sends non errors to stderr (stupid)
        // @see https://stackoverflow.com/questions/13066518/why-does-wget-output-to-stderr-rather-than-stdout
        if (is_file($errs)) {
            if ($errors !== '') {
                if ($errs = jrCore_file_get_contents($errs)) {
                    if (!strpos($errs, ' -> ')) {
                        // 2023-08-04 07:05:28 URL:https://www.nbcnews.com/news/variants-omicron-develop-makes-variants-concern-rcna6798 [72096] -> "/Users/brian/Web/Master/data/cache/jrUrlScan/8k0br4qx4ou6" [1]
                        $errors = trim(str_replace("{$url}:\n", "", $errs));
                    }
                }
            }
            jrCore_unlink($errs);
        }

        if (is_file($temp)) {
            if ($cookies !== '') {
                $cookies = jrUrlScan_get_cookie_array(jrCore_file_get_contents($temp));
            }
            jrCore_unlink($temp);
        }

        if (is_file($file)) {
            $temp = jrCore_file_get_contents($file);
            jrCore_unlink($file);
            if (function_exists('gzdecode')) {
                if ($test = @gzdecode($temp)) {
                    // We were encoded
                    if (stripos(' ' . $test, '<head')) {
                        return $test;
                    }
                }
            }
            // Make sure we got HTML
            if (stripos(' ' . $temp, '<head')) {
                return $temp;
            }
        }
    }
    return false;
}

/**
 * Get site cookies in to an accessible array
 * @param string $cookies
 * @return array|bool
 */
function jrUrlScan_get_cookie_array($cookies)
{
    if (!empty($cookies)) {
        $_result = array();
        $cookies = trim($cookies);
        foreach (explode("\n", $cookies) as $l) {
            // domain          flag    path    secure  expires         name                    value
            // github.com      FALSE   /       FALSE   1550851146      has_recent_activity     1
            if (strpos($l, 'TRUE') || strpos($l, 'FALSE')) {
                $_result[] = array(
                    'domain'  => jrCore_string_field($l, 1),
                    'flag'    => jrCore_string_field($l, 2),
                    'path'    => jrCore_string_field($l, 3),
                    'secure'  => jrCore_string_field($l, 4),
                    'expires' => jrCore_string_field($l, 5),
                    'name'    => jrCore_string_field($l, 6),
                    'value'   => jrCore_string_field($l, 7)
                );
            }
        }
        return $_result;
    }
    return false;
}

/**
 * Get the active user agent
 * @return string
 */
function jrUrlScan_get_user_agent()
{
    // Are we being overridden?
    if ($agent = jrCore_get_config_value('jrUrlScan', 'user_agent', false)) {
        return $agent;
    }
    $is_bot = jrUser_get_bot_name();
    if (!empty($is_bot)) {
        // Viewing user is a BOT - do not use bot agent
        return JRURLSCAN_USER_AGENT;
    }
    return (!empty($_SERVER['HTTP_USER_AGENT'])) ? $_SERVER['HTTP_USER_AGENT'] : JRURLSCAN_USER_AGENT;
}

/**
 * Save an image for a URL card
 * @param $item_id int UrlScan item id for URL
 * @param $image_url string URL to image
 * @param $profile_id int Profile ID
 * @param null $_item array (optional) Item info
 * @return bool
 */
function jrUrlScan_save_url_image($item_id, $image_url, $profile_id, $_item = null)
{
    $dir = jrCore_get_module_cache_dir('jrUrlScan');
    $tgt = "{$dir}/jrUrlScan_{$item_id}_urlscan_image.tmp";
    if (!jrCore_download_file($image_url, $tgt, 120, 80, null, null, null, false)) {
        if ($fgc = jrCore_file_get_contents($image_url)) {
            jrCore_write_to_file($tgt, $fgc);
        }
        else {
            return false;
        }
    }
    // Figure out our image type
    if ($ext = jrCore_file_extension_from_mime_type(jrCore_mime_type($tgt))) {
        switch ($ext) {
            case 'jfif':
            case 'jpeg':
            case 'jpe':
                $ext = 'jpg';
                break;
        }
        if (!jrImage_is_image_file("file.{$ext}")) {
            // Bad image image
            jrCore_unlink($tgt);
            return false;
        }
        $new = str_replace('.tmp', ".{$ext}", $tgt);
        if (!rename($tgt, $new)) {
            return false;
        }
        $tgt = $new;
    }
    else {
        // Could not get image
        jrCore_unlink($tgt);
        return false;
    }
    jrCore_save_media_file('jrUrlScan', $tgt, $profile_id, $item_id, 'urlscan_image');
    jrCore_unlink($tgt);
    return true;
}

/**
 * Create a new card for a URL
 * @param string $url URL
 * @param int $profile_id Profile ID creating card
 * @param bool $check_if_exists Set to FALSE to not check if URL already exists in DS
 * @return array|false
 */
function jrUrlScan_create_url_card($url, $profile_id, $check_if_exists = true)
{
    if (!jrCore_checktype($url, 'url')) {
        return false;
    }
    if (jrUrlScan_is_local_player_url($url)) {
        return false;
    }
    // Does this URL already exist?
    if ($check_if_exists) {
        if ($_it = jrCore_db_get_item_by_key('jrUrlScan', 'urlscan_url', $url, true)) {
            // The URL card for this URL has already been created
            return $_it;
        }
    }
    $_cr = array('_profile_id' => $profile_id);
    $tmp = jrUrlScan_get_safe_url($url);
    if ($tmp && is_array($tmp)) {
        // We have a malware URL
        $_rp = array(
            'location' => jrCore_get_current_url(),
            'response' => $tmp
        );
        jrCore_logger('MAJ', "urlscan: URL has been flagged as unsafe: {$url}", $_rp);
        $_item = array(
            'urlscan_url'        => $url,
            'urlscan_unsafe'     => 1,
            'urlscan_view_count' => 1
        );
        $uid   = jrCore_db_create_item('jrUrlScan', $_item, $_cr);
        return jrCore_db_get_item('jrUrlScan', $uid, true, true);
    }
    else {
        $url = jrUrlScan_get_trimmed_url($url);
    }
    // @note: we can $skip_check here since we test for a local player
    // at the very top of this function with jrUrlScan_is_local_player_url()
    $_tags = jrUrlScan_extract_tags_from_url($url, null, true);
    if ($_tags && is_array($_tags) && count($_tags) > 0) {

        $_tags = jrUrlScan_validate_tags($_tags);
        $_tags = jrUrlScan_get_url_tags_to_save($url, $_tags);
        if ($_tags && is_array($_tags)) {

            // If this page does not have an OG image and we are using headless Chrome
            // We can use Chrome to snap a screenshot of the page and use that
            if (empty($_tags['urlscan_ogimage'])) {
                // We set the image_size = 1 so the "default" image will show
                $_tags['urlscan_image_size'] = 1;
            }

            $uid = jrCore_db_create_item('jrUrlScan', $_tags, $_cr);
            if ($uid && jrCore_checktype($uid, 'number_nz')) {
                if (isset($_tags['urlscan_ogimage']) && jrCore_checktype($_tags['urlscan_ogimage'], 'url')) {
                    if (!jrUrlScan_save_url_image($uid, $_tags['urlscan_ogimage'], $profile_id, $_tags)) {
                        // We had an error saving the image - go through our tags and see if we have
                        // any alternate image tags for this URL
                        foreach ($_tags as $k => $v) {
                            if (strpos($k, 'image') && $k != 'urlscan_ogimage' && jrCore_checktype($v, 'url')) {
                                if (jrUrlScan_save_url_image($uid, $v, $profile_id, $_tags)) {
                                    // We saved it
                                    break;
                                }
                            }
                        }
                    }
                }

            }
            return jrCore_db_get_item('jrUrlScan', $uid, true, true);
        }
        // Fall through - URL is missing title - URL is still good
    }

    // Fall through - create entry for URL
    $_item = array(
        'urlscan_url' => $url
    );
    if ($uid = jrCore_db_create_item('jrUrlScan', $_item, $_cr)) {
        $_item['_item_id'] = $uid;
        return $_item;
    }
    return false;
}

/**
 * Refresh a url card
 * @param string $item_id URL
 * @return array|false
 */
function jrUrlScan_refresh_url_card($item_id)
{
    $_it = jrCore_db_get_item('jrUrlScan', $item_id);
    if (!$_it) {
        return false;
    }

    $url        = $_it['urlscan_url'];
    $profile_id = $_it['_profile_id'];

    $_cr = array('_profile_id' => $profile_id);
    $tmp = jrUrlScan_get_safe_url($url);
    if ($tmp && is_array($tmp)) {
        // We have a malware URL
        $_rp = array(
            'location' => jrCore_get_current_url(),
            'response' => $tmp
        );
        jrCore_logger('MAJ', "urlscan: URL has been flagged as unsafe: {$url}", $_rp);
        $_sv = array(
            'urlscan_unsafe' => 1
        );
        jrCore_db_update_item('jrUrlScan', $item_id, $_sv, $_cr);
        return jrCore_db_get_item('jrUrlScan', $item_id, true, true);
    }
    else {
        $url = jrUrlScan_get_trimmed_url($url);
    }
    // @note: we can $skip_check here since we test for a local player
    // at the very top of this function with jrUrlScan_is_local_player_url()
    $_tags = jrUrlScan_extract_tags_from_url($url, null, true);
    if ($_tags && is_array($_tags) && count($_tags) > 0) {

        $_tags = jrUrlScan_validate_tags($_tags);
        $_tags = jrUrlScan_get_url_tags_to_save($url, $_tags);
        if ($_tags && is_array($_tags)) {

            // If this page does not have an OG image and we are using headless Chrome
            // We can use Chrome to snap a screenshot of the page and use that
            if (empty($_tags['urlscan_ogimage'])) {
                // We set the image_size = 1 so the "default" image will show
                $_tags['urlscan_image_size'] = 1;
            }

            jrCore_db_update_item('jrUrlScan', $item_id, $_tags, $_cr);
            if (isset($_tags['urlscan_ogimage']) && jrCore_checktype($_tags['urlscan_ogimage'], 'url')) {
                if (!jrUrlScan_save_url_image($item_id, $_tags['urlscan_ogimage'], $profile_id, $_tags)) {
                    // We had an error saving the image - go through our tags and see if we have
                    // any alternate image tags for this URL
                    foreach ($_tags as $k => $v) {
                        if (strpos($k, 'image') && $k != 'urlscan_ogimage' && jrCore_checktype($v, 'url')) {
                            if (jrUrlScan_save_url_image($item_id, $v, $profile_id, $_tags)) {
                                // We saved it
                                break;
                            }
                        }
                    }
                }
            }
            return jrCore_db_get_item('jrUrlScan', $item_id, true, true);
        }
        // Fall through - URL is missing title - URL is still good
    }
    return false;
}

/**
 * Parse the tags and get uniform tag names
 * @param string $url
 * @param array $_tags
 * @return array|bool
 */
function jrUrlScan_get_url_tags_to_save($url, $_tags)
{
    if (!isset($_tags['og:title'])) {
        // This could be a URL that has no IG tags - in that
        // case we're going to check for TITLE and DESCRIPTION
        // and use those if we can
        if (!empty($_tags['title'])) {
            $_tags['og:title'] = $_tags['title'];
            unset($_tags['title']);
        }
        elseif (!empty($_tags['description'])) {
            $_tags['og:title']       = 'no title';
            $_tags['og:description'] = $_tags['description'];
            unset($_tags['description']);
        }
        if (!isset($_tags['og:title'])) {
            // We don't know the URL title, so we're not going to be able to
            // show a nice card or anything - return FALSE so we just use the URL
            return false;
        }
    }
    $_rt = array(
        'urlscan_url'        => jrUrlScan_get_trimmed_url($url),
        'urlscan_title'      => $_tags['og:title'],
        'urlscan_title_url'  => jrCore_url_string($_tags['og:title']),
        'urlscan_view_count' => 1
    );
    foreach ($_tags as $k => $v) {
        switch ($k) {
            case 'og:sitename':
            case 'og:site_name':
                $tag       = 'urlscan_ogsitename';
                $_rt[$tag] = $v;
                break;
            case 'og:image':
            case 'og:description':
            case 'og:keywords':
                $tag       = 'urlscan_og' . substr($k, 3);
                $_rt[$tag] = $v;
                break;
            default:
                if (strpos($k, 'image')) {
                    $tag       = 'urlscan_og' . str_replace(array(':', '.'), '_', substr($k, 3));
                    $_rt[$tag] = $v;
                }
                break;
        }
    }
    return $_rt;
}

/**
 * Update an existing URL Card
 * @param $_url array existing DS entry for URL
 * @return bool
 */
function jrUrlScan_update_url_card($_url)
{
    if (!is_array($_url) || !isset($_url['urlscan_url'])) {
        return false;
    }
    $_tags = jrUrlScan_extract_tags_from_url($_url['urlscan_url']);
    if ($_tags && is_array($_tags) && count($_tags) > 0) {

        // We have successfully parsed this URL
        // Remove any keys that had previously indicated otherwise
        if (isset($_url['urlscan_timed_out']) || isset($_url['urlscan_not_found'])) {
            jrCore_db_delete_multiple_item_keys('jrUrlScan', $_url['_item_id'], array('urlscan_timed_out', 'urlscan_not_found'));
        }

        $_tags = jrUrlScan_validate_tags($_tags);
        $_tags = jrUrlScan_get_url_tags_to_save($_url['urlscan_url'], $_tags);

        if ($_tags && is_array($_tags)) {
            if (jrCore_db_update_item('jrUrlScan', $_url['_item_id'], $_tags)) {
                if (empty($_url['urlscan_image_size'])) {
                    // We do not have an image for this entry - see if we can get one
                    if (isset($_tags['urlscan_ogimage']) && jrCore_checktype($_tags['urlscan_ogimage'], 'url')) {
                        jrUrlScan_save_url_image($_url['_item_id'], $_tags['urlscan_ogimage'], $_url['_profile_id'], $_url);
                    }
                }
                return true;
            }
            else {
                return false;
            }
        }
    }
    elseif (is_array($_tags)) {
        // URL is good, but we did not get HTML back to create tags
        if (isset($_url['urlscan_timed_out']) || isset($_url['urlscan_not_found'])) {
            jrCore_db_delete_multiple_item_keys('jrUrlScan', $_url['_item_id'], array('urlscan_timed_out', 'urlscan_not_found'));
            return false;
        }
    }

    // We have an error condition for this URL
    switch ($_tags) {
        case 404:
            // This site is no longer found - mark as dead
            jrCore_db_update_item('jrUrlScan', $_url['_item_id'], array('urlscan_not_found' => 1));
            break;
        case 'retry':
            if (isset($_url['urlscan_timed_out']) && jrCore_checktype($_url['urlscan_timed_out'], 'number_nz')) {
                // It looks like this URL has already been timing out - how long?
                $diff = (time() - $_url['urlscan_timed_out']);
                if (($diff / 86400) >= 7) {
                    // We've gone 7 days timing out - mark as dead
                    jrCore_db_update_item('jrUrlScan', $_url['_item_id'], array('urlscan_not_found' => 1));
                }
            }
            else {
                jrCore_db_update_item('jrUrlScan', $_url['_item_id'], array('urlscan_timed_out' => 'UNIX_TIMESTAMP()'));
            }
            break;
    }
    return false;
}

/**
 * Validate and re-map tags
 * @param $_tags array
 * @return array|false
 */
function jrUrlScan_validate_tags($_tags)
{
    // @note This function will go through the meta tags from the
    // page and make sure we only come out with og: tags
    if (is_array($_tags)) {
        foreach ($_tags as $k => $v) {
            if (strpos($k, ':') && strpos($k, 'og:') !== 0) {
                list(, $tag) = explode(':', $k);
                if (!isset($_tags["og:{$tag}"])) {
                    $_tags["og:{$tag}"] = $v;
                }
                else {
                    // If this is an IMAGE then we want to save it
                    // in case the regular og: image is incorrect/bad
                    if (strpos($k, 'image') && jrCore_checktype($v, 'url')) {
                        if ($v != $_tags['og:image']) {
                            $t                = str_replace(':', '', $k);
                            $_tags["og:{$t}"] = $v;
                        }
                    }
                }
                unset($_tags[$k]);
            }
        }
        return $_tags;
    }
    return false;
}

/**
 * Get card for a URL
 * @param string $url URL to create card for
 * @param int $profile_id Profile to create card for (0 = use active user profile id)
 * @return array|false
 */
function jrUrlScan_get_url_card($url, $profile_id = 0)
{
    global $_user;
    if (jrCore_checktype($url, 'url')) {

        // If this is a local PLAYER we don't hand;e that here
        if (jrUrlScan_is_local_player_url($url)) {
            return false;
        }

        $url = jrUrlScan_get_trimmed_url($url);
        if (!$_it = jrCore_db_get_item_by_key('jrUrlScan', 'urlscan_url', $url, true)) {
            $pid = (int) $profile_id;
            if ($pid === 0) {
                if (!empty($_user['user_active_profile_id'])) {
                    $pid = (int) $_user['user_active_profile_id'];
                }
            }
            if ($_it = jrUrlScan_create_url_card($url, $pid, false)) {
                if (is_array($_it)) {
                    if (empty($_it['urlscan_ogtitle']) && !empty($_it['urlscan_title'])) {
                        $_it['urlscan_ogtitle'] = $_it['urlscan_title'];
                    }
                    elseif (empty($_it['urlscan_ogtitle']) && !empty($_it['urlscan_description'])) {
                        $_it['urlscan_ogtitle'] = $_it['urlscan_description'];
                    }
                    return $_it;
                }
                // Fall through - URL is either 404 or timed out
            }
            // We could not save this one - return URL
            return array('urlscan_url' => $url);
        }
        else {
            // Increment view and updated values - this way we
            // can keep track of how frequently a URL needs to be kept up to date
            jrCore_db_increment_key('jrUrlScan', $_it['_item_id'], 'urlscan_view_count', 1, true);
        }
        if (empty($_it['urlscan_ogtitle']) && !empty($_it['urlscan_title'])) {
            $_it['urlscan_ogtitle'] = $_it['urlscan_title'];
        }
        return $_it;

    }
    return false;
}

/**
 * Check if a given URL is a local URL with a URLScan player template
 * @param $url string
 * @return bool
 */
function jrUrlScan_is_local_player_url($url)
{
    global $_urls, $_conf;
    // Is this a LOCAL URL that has a URL scan player template?
    if (strpos($url, $_conf['jrCore_base_url']) === 0) {
        // Local
        $crl = trim(rtrim($_conf['jrCore_base_url'], '/'));
        $_tm = explode('/', str_replace("{$crl}/", '', $url));
        if ($_tm && is_array($_tm)) {
            if (isset($_tm[1]) && isset($_urls["{$_tm[1]}"]) && isset($_tm[2]) && jrCore_checktype($_tm[2], 'number_nz')) {
                // We have a local item
                $mod = $_urls["{$_tm[1]}"];
                if (is_file(APP_DIR . "/modules/{$mod}/templates/urlscan_player.tpl")) {
                    // This is handled by the module
                    return true;
                }
            }
        }
    }
    return false;
}

/**
 * Get additional info about a URL?
 * @param $_data array
 * @param $_args array
 * @return array|false
 */
function jrUrlScan_get_card_info_for_url($_data, $_args)
{
    global $_conf;
    if (isset($_args['url']) && !jrUrlScan_is_local_player_url($_args['url'])) {
        $_it = jrUrlScan_get_url_card($_args['url']);
        if ($_it && is_array($_it)) {

            if (isset($_it['urlscan_not_found']) && $_it['urlscan_not_found'] == 1) {
                // This URL is NOT found
                $_data['_items'][$_args['i']]['title']     = 'not_found';
                $_data['_items'][$_args['i']]['not_found'] = 1;
            }
            elseif (isset($_it['urlscan_unsafe']) && $_it['urlscan_unsafe'] == 1) {
                // This is an UNSAFE URL
                $_data['_items'][$_args['i']]['title']  = 'unsafe';
                $_data['_items'][$_args['i']]['unsafe'] = 1;
            }
            elseif (isset($_it['urlscan_title'])) {
                // This is a GOOD URL
                $uurl                                     = jrCore_get_module_url('jrUrlScan');
                $_data['_items'][$_args['i']]['title']    = $_it['urlscan_title'];
                $_data['_items'][$_args['i']]['load_url'] = "{$_conf['jrCore_base_url']}/{$uurl}/ogtags/{$_it['_item_id']}/__ajax=1";
            }
            $_data['_items'][$_args['i']]['url']   = $_args['url'];
            $_data['_items'][$_args['i']]['_item'] = $_it;
        }
        return $_data;
    }
    return false;
}

/**
 * Check if a given URL is safe via Google's Safe Browsing API
 * @param string $url
 * @return string
 */
function jrUrlScan_get_safe_url($url)
{
    $_urls = array($url);
    if ($_res = jrUrlScan_bulk_check_safe_urls($_urls)) {
        if (!empty($_res['unsafe'][0])) {
            return $_res['unsafe'];
        }
    }
    return $url;
}

/**
 * Check if a given URL is safe via Google's Safe Browsing API
 * @param array $_urls
 * @return array|false
 */
function jrUrlScan_bulk_check_safe_urls($_urls)
{
    global $_mods, $_conf;
    if (!empty($_conf['jrUrlScan_api_key']) && isset($_conf['jrUrlScan_safe_urls']) && $_conf['jrUrlScan_safe_urls'] == 'on') {
        $tmp  = '{
            "client": {
                "clientId": "' . $_conf['jrCore_system_name'] . '",
                "clientVersion": "' . $_mods['jrCore']['module_version'] . '"
            },
            "threatInfo": {
                "threatTypes":      ["MALWARE", "SOCIAL_ENGINEERING"],
                "platformTypes":    ["ALL_PLATFORMS"],
                "threatEntryTypes": ["URL"],
                "threatEntries": [
                    {"url": "' . implode('"},{"url": "', $_urls) . '"}
                ]
            }
        }';
        $_res = array(
            'unsafe' => array(),
            'valid'  => array()
        );
        $ch   = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://safebrowsing.googleapis.com/v4/threatMatches:find?key={$_conf['jrUrlScan_api_key']}");
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array("Content-Type: application/json", 'Content-Length: ' . strlen($tmp)));
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $tmp);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        $res = json_decode(curl_exec($ch), true);
        curl_close($ch);
        if ($res && is_array($res) && isset($res['matches'])) {
            foreach ($res['matches'] as $m) {
                $_res['unsafe'][] = $m['threat']['url'];
            }
            foreach ($_urls as $k => $url) {
                if (in_array($url, $_res['unsafe'])) {
                    unset($_urls[$k]);
                }
            }
        }
        if (count($_urls) > 0) {
            $_res['valid'] = $_urls;
        }
        return $_res;
    }
    return false;
}

/**
 * Strip common tracking parameters from a URL
 * @param string $url
 * @return string
 */
function jrUrlScan_strip_url_tracking_params($url)
{
    $_trackers = array(
        'utm_source',
        'utm_medium',
        'utm_term',
        'utm_content',
        'utm_campaign',
        'utm_reader',
        'utm_place',
        'utm_userid',
        'utm_cid',
        'utm_name',
        'utm_pubreferrer',
        'utm_swu',
        'utm_viz_id',
        'utm_mailing',
        'utm_brand',
        'ga_source',
        'ga_medium',
        'ga_term',
        'ga_content',
        'ga_campaign',
        'ga_place',
        'yclid',
        '_openstat',
        'fb_action_ids',
        'fb_action_types',
        'fb_ref',
        'fb_source',
        'action_object_map',
        'action_type_map',
        'action_ref_map',
        'gs_l',
        '_hsenc',
        'mkt_tok',
        'hmb_campaign',
        'hmb_source',
        'hmb_medium',
        'fbclid',
        'spReportId',
        'spJobID',
        'spUserID',
        'spMailingID',
        'CNDID',
        'mbid',
        'trk',
        'trkCampaign',
        'sc_campaign',
        'sc_channel',
        'sc_content',
        'sc_medium',
        'sc_outcome',
        'sc_geo',
        'sc_country'
    );
    return jrCore_strip_url_params($url, $_trackers);
}

/**
 * Replace URLs in text with player
 * @param $text string
 * @return string
 */
function smarty_modifier_jrUrlScan_replace_urls($text)
{
    $text = jrUrlScan_replace_urls($text);
    if ($_rp = jrCore_get_flag(JRURLSCAN_TEXT_KEY)) {
        $text = str_replace(array_keys($_rp), $_rp, $text);
        jrCore_delete_flag(JRURLSCAN_TEXT_KEY);
    }
    if ($_rp = jrCore_get_flag(JRURLSCAN_URLS_KEY)) {
        $text .= "\n" . '<script type="text/javascript">$(document).ready(function(){' . implode('', $_rp) . '});</script>';
        jrCore_delete_flag(JRURLSCAN_URLS_KEY);
    }
    return $text;
}
