<?php
 /**
 * Jamroom Users module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2021 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * view: account_save
 * @param array $_post Posted Data
 * @param array $_user Viewing User data
 * @param array $_conf Global Config
 */
function view_jrUser_account_save($_post, $_user, $_conf)
{
    jrUser_session_require_login();
    jrCore_form_validate($_post);

    // Double check email
    if (!isset($_post['user_email']) || !jrCore_checktype($_post['user_email'], 'email')) {
        jrCore_set_form_notice('error', 68);
        jrCore_form_field_hilight('user_email');
        jrCore_form_result();
    }

    // User name checks
    if (!jrUser_is_admin() && !empty($_post['user_name'])) {
        if (!jrCore_checktype($_post['user_name'], 'valid_name')) {
            jrCore_set_form_notice('error', 129);
            jrCore_form_field_hilight('user_name');
            jrCore_form_result();
        }
    }

    // Get posted data
    $rauth = false;
    $_data = jrCore_form_get_save_data('jrUser', 'account', $_post);

    // Check for changing passwords
    if ((isset($_data['user_passwd1']) && strlen($_data['user_passwd1']) > 0) || (isset($_data['user_passwd2']) && strlen($_data['user_passwd2']) > 0)) {
        if (isset($_data['user_passwd1']) && isset($_data['user_passwd2']) && $_data['user_passwd1'] != $_data['user_passwd2']) {
            jrCore_set_form_notice('error', 35);
            jrCore_form_field_hilight('user_passwd1');
            jrCore_form_field_hilight('user_passwd2');
            jrCore_form_result();
        }
        // Add in new password hash
        $pass                   = jrUser_get_password_hash($_data['user_passwd1']);
        $_data['user_password'] = $pass;
    }

    // Check for forced re-authentication
    if (isset($_conf['jrUser_authenticate']) && $_conf['jrUser_authenticate'] == 'on') {

        if (!isset($_data['user_id']) || !jrCore_checktype($_data['user_id'], 'number_nz')) {
            $_data['user_id'] = $_user['_user_id'];
        }

        if (jrUser_is_admin() && isset($_data['user_id']) && jrCore_checktype($_data['user_id'], 'number_nz') && $_data['user_id'] != $_user['_user_id']) {
            // Admin changing another account
            $rauth = false;
        }
        else {

            // Forced Password change
            // NOTE: This is set in the OneAll module to force the user to set a NEW password after account creation
            if (isset($_user['user_temp_password']) && !isset($_user['user_temp_token'])) {
                $rauth = false;
            }
            else {
                // Password if we are changing
                if (isset($pass)) {
                    $_data['user_temp_password'] = $pass;
                    unset($_data['user_password']);
                    $rauth = true;
                }

                // Email changing
                if ($_data['user_email'] != $_user['user_email']) {
                    $_data['user_temp_email'] = $_data['user_email'];
                    unset($_data['user_email']);
                    $rauth = true;
                }

                // If 2 factor auth is being turned OFF
                if (jrCore_module_is_active('jrTwoFactor') && isset($_user['user_twofactor_enabled']) && $_user['user_twofactor_enabled'] == 'on' && $_data['user_twofactor_enabled'] == 'off') {
                    $_data['user_temp_twofactor_enabled'] = $_data['user_twofactor_enabled'];
                    unset($_data['user_twofactor_enabled']);
                    $rauth = true;
                }

                if ($rauth) {
                    $_data['user_temp_token'] = $_post['jr_html_form_token'];
                }
            }

        }
    }

    // See if this is an admin modifying this user account
    $uid = $_user['_user_id'];
    if (jrUser_is_admin() && isset($_data['user_id']) && jrCore_checktype($_data['user_id'], 'number_nz')) {

        $uid = (int) $_data['user_id'];
        $_us = jrCore_db_get_item('jrUser', $uid);
        if (!empty($_data['user_group'])) {
            $_data['user_group'] = $_data['user_group'];
        }
        // Check for changes in linked profiles
        if (isset($_data['user_linked_profiles']) && strlen($_data['user_linked_profiles']) > 0) {
            $pid = (int) $_post['profile_id'];
            $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
            if ($_data['user_linked_profiles'] == '_') {
                // user_linked_profiles will be an underscore if ALL extra profiles are removed
                $req = "DELETE FROM {$tbl} WHERE user_id = '{$uid}' AND profile_id != '{$pid}'";
            }
            else {
                $_lp = array();
                foreach (explode(',', $_data['user_linked_profiles']) as $lpid) {
                    if (jrCore_checktype($lpid, 'number_nz')) {
                        $_lp[$lpid] = $lpid;
                    }
                }
                if (count($_lp) > 0) {
                    $req = "DELETE FROM {$tbl} WHERE user_id = {$uid} AND profile_id NOT IN({$pid}," . implode(',', $_lp) . ")";
                }
                else {
                    $req = "DELETE FROM {$tbl} WHERE user_id = {$uid} AND profile_id != {$pid}";
                }
            }
            jrCore_db_query($req);
        }
    }
    else {
        $_us = $_user;
    }
    // We don't save user_linked_profiles to the DS
    if (isset($_data['user_linked_profiles'])) {
        unset($_data['user_linked_profiles']);
    }

    // Check for changing user_email
    if (!empty($_data['user_temp_email'])) {
        $_sc = array(
            'search'         => array(
                "user_email = {$_data['user_temp_email']}",
                "_item_id != {$uid}"
            ),
            'return_count'   => true,
            'skip_triggers'  => true,
            'privacy_check'  => false,
            'ignore_missing' => true,
            'ignore_pending' => true
        );
        $cnt = jrCore_db_search_items('jrUser', $_sc);
        if ($cnt && $cnt > 0) {
            $_SESSION['show_email_exists_error'] = true;
            jrCore_set_form_notice('error', 96);
            jrCore_form_field_hilight('user_email');
            jrCore_form_result();
        }
    }

    // Check for changing user_name
    if (!empty($_data['user_name'])) {
        $_sc = array(
            'search'         => array(
                "user_name = {$_data['user_name']}",
                "_item_id != {$uid}"
            ),
            'return_count'   => true,
            'skip_triggers'  => true,
            'privacy_check'  => false,
            'ignore_missing' => true,
            'ignore_pending' => true
        );
        $cnt = jrCore_db_search_items('jrUser', $_sc);
        if ($cnt && $cnt > 0) {
            jrCore_set_form_notice('error', 100);
            jrCore_form_field_hilight('user_name');
            jrCore_form_result();
        }
    }

    // See if we got a language
    if (jrUser_is_admin() && !empty($_data['user_blocked'])) {
        if ($_data['user_blocked'] == 'on') {
            $_data['user_active']  = 0;
            $_data['user_blocked'] = 1;
        }
        else {
            jrCore_db_delete_item_key('jrUser', $uid, 'user_blocked', false, false);
            $_data['user_active'] = 1;
            unset($_data['user_blocked']);
        }
    }

    // Save info
    unset($_data['user_passwd1'], $_data['user_passwd2']);
    jrCore_db_update_item('jrUser', $uid, $_data);

    // If we are changing email or password, send an email to the OLD email address
    // outlining that the email address has been changed on the account
    if (!jrUser_is_admin() && (!isset($_conf['jrUser_authenticate']) || $_conf['jrUser_authenticate'] != 'on')) {
        if (isset($_conf['jrUser_change_notice']) && $_conf['jrUser_change_notice'] == 'on') {
            $_rp = array(
                'system_name'     => $_conf['jrCore_system_name'],
                'password_change' => false,
                'email_change'    => false
            );
            // Is the user changing email?
            if ($_data['user_email'] != $_user['user_email']) {
                $_rp['email_change'] = true;
            }
            if (isset($pass)) {
                $_rp['password_change'] = true;
            }
            list($sub, $msg) = jrCore_parse_email_templates('jrUser', 'change', $_rp);
            jrCore_send_email($_user['user_email'], $sub, $msg);
        }
    }

    // Check for Photo upload
    $tempid = $_us['_profile_id'];
    $_image = jrCore_save_media_file('jrUser', 'user_image', $_us['_profile_id'], $uid);
    // If the user does NOT have a profile image yet, set the user image to be the profile image...
    if (!isset($_us['profile_image_size']) && isset($_image) && is_array($_image)) {
        $_us        = array_merge($_us, $_image);
        $user_image = jrCore_get_media_file_path('jrUser', 'user_image', $_us);
        if (is_file($user_image)) {
            $ext = jrCore_file_extension($user_image);
            $nam = "{$_us['_profile_id']}_profile_image";
            if (jrCore_copy_media_file($_us['_profile_id'], $user_image, $nam)) {
                $dir = dirname($user_image);
                jrCore_write_to_file("{$dir}/{$nam}.tmp", "profile_image.{$ext}");
                jrCore_save_media_file('jrProfile', "{$dir}/{$nam}", $_us['_profile_id'], $_us['_profile_id']);
                unlink("{$dir}/{$nam}");
                unlink("{$dir}/{$nam}.tmp");
            }
        }
        $_us['_profile_id'] = $tempid;
    }
    unset($tempid);

    // Remove temp password flag if set
    if (!$rauth && isset($_user['user_temp_password'])) {
        jrCore_db_delete_item_key('jrUser', $_user['_user_id'], 'user_temp_password');
        jrUser_delete_session_key('user_temp_password');
    }

    // Re-sync session
    if (isset($uid) && $uid == $_user['_user_id']) {
        jrUser_reset_cache($uid);
        jrUser_session_sync();
    }

    // Reset caches
    $_ln = jrProfile_get_user_linked_profiles($_us['_user_id']);
    if ($_ln && is_array($_ln)) {
        foreach ($_ln as $pid => $uid) {
            jrProfile_reset_cache($pid);
        }
    }

    // If we are an ADMIN user modifying someone else...
    if (jrUser_is_admin() && isset($_data['user_id']) && jrCore_checktype($_data['user_id'], 'number_nz') && $_data['user_id'] != $_user['_user_id']) {

        // Send out account updated trigger
        jrCore_trigger_event('jrUser', 'user_updated', $_us, $_data);
        jrUser_set_session_sync_for_user_id($_data['user_id'], 'on');

        jrCore_set_form_notice('success', 'The user account has been successfully updated');
        // If this is an admin from the browser...
        if (isset($_post['from_browser']) && jrCore_checktype($_post['from_browser'], 'url')) {
            jrCore_form_result($_post['from_browser']);
        }
        jrCore_form_delete_session();
        jrCore_form_result();
    }

    // Fall through - we are modifying our own account
    if ($rauth) {
        jrCore_form_delete_session();
        jrCore_form_result("{$_conf['jrCore_base_url']}/{$_post['module_url']}/authenticate");
    }

    // Send out account updated trigger
    jrCore_trigger_event('jrUser', 'user_updated', $_us, $_data);

    // If we changed passwords, log out other sessions
    if (!empty($pass)) {
        jrUser_session_remove_all_other_sessions($_user['_user_id'], session_id());
        if (jrCore_get_cookie('auto')) {
            // This user had a remember me cookie - set new one
            jrUser_session_set_login_cookie($_user['_user_id']);
        }
    }

    // Remove any forgot password entries
    jrUser_delete_forgot_password_entries($_user['_user_id']);

    jrCore_set_form_notice('success', 43);
    jrCore_form_delete_session();
    jrUser_reset_cache($_user['_user_id']);
    jrCore_form_result();
}
